/**
 * $Id: ParentDirTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.mktemp;

import  java.io.File;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.apis.AntLibFriendly;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.LocalTk;
import  com.idaremedia.antx.parameters.FlexExistenceEnabled;

/**
 * Task that extracts and saves the parent directory of a file system object. Currently
 * the &lt;parentdir&gt; task handles files and file URLs as well as two internal
 * pseudo-URL types: <span class="src">resource:</span> and <span class="src">tmp:</span>.
 * See next section for some examples.
 * <p>
 * If both the "<span class="src">mustexist</span>" and "<span class="src">append</span>"
 * parameters are specified, the existence check is performed on the calculated
 * parent directory <em>before</em> the subpath bits are appended.
 * <p>
 * <b>Example Usage:</b><pre>
 *    &lt;tempdir urlproperty="TMP"/&gt;
 *    &lt;<b>parentdir</b> of="${TMP}" urlproperty="TMP.."/&gt;
 * 
 *    &lt;<b>parentdir</b> pop="2" pathproperty="basedir/../.."/&gt;
 *    &lt;<b>parentdir</b> of="resource:antx/oofs/test.properties" pathproperty="ETC"/&gt;
 *    &lt;<b>parentdir</b> of="${nextbuild.file}" mustexist="yes" pathproperty="nextroot"/&gt;
 * 
 *    &lt;<b>parentdir</b> of="${basedir}" pop="2" pathproperty="etc.dir" append="etc"/>
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class ParentDirTask extends AssertableTask
    implements FlexExistenceEnabled, AntLibFriendly
{
    /**
     * Initializes a new parent dir task.
     **/
    public ParentDirTask()
    {
        super(AntX.mktemp+"ParentDirTask:");
    }


    /**
     * Sets the file (by path or by URL) that we'd like the
     * parent of. File URLs are converted to file system-dependent
     * paths before being processed.
     * @param fileref the file reference (non-null)
     **/
    public void setOf(String fileref)
    {
        require_(fileref!=null,"setOf- nonzro path descriptor");
        m_fileref = fileref;
    }



    /**
     * Returns the file (path or URL) that we'll try to get the
     * parent of. Will return <i>null</i> if never set explicitly.
     **/
    public String getOf()
    {
        return m_fileref;
    }



    /**
     * Sets the property to be updated with the parent directory's
     * full path.
     * @param property property to be updated (non-null)
     **/
    public void setPathProperty(String property)
    {
        require_(property!=null,"setProp- nonzro nam");
        m_updateProperty = property;
    }



    /**
     * Returns the property updated with the parent directory's
     * full path. Returns <i>null</i> if never set.
     **/
    public String getPathProperty()
    {
        return m_updateProperty;
    }



    /**
     * Sets the property into which a URL representation of the 
     * parent directory's path is copied.
     * @param urlproperty the property's name (non-null)
     **/
    public void setUrlProperty(String urlproperty)
    {
        require_(urlproperty!=null,"setUrlProp- nonzro name");
        m_updateUrlProperty = urlproperty;
    }



    /**
     * Returns the name of the property into which a URL representation
     * of the parent directory's path will be stored. Returns <i>null</i>
     * if never set explicitly.
     **/
    public final String getUrlPathProperty()
    {
        return m_updateUrlProperty;
    }



    /**
     * Tells this task whether the parent directory must exist or not.
     * @param mustExist <i>true</i> if parent must exist.
     *
     */
    public void setMustExist(boolean mustExist)
    {
        m_mustExist = mustExist;
    }



    /**
     * Returns <i>true</i> if the calculated parent directory must
     * already exist. Defaults to <i>false</i>.
     */
    public boolean getMustExist()
    {
        return m_mustExist;
    }



    /**
     * Gives this task a specific number of "pop" level to create
     * parent or grand-parent operation.
     * @param levelDesc levels to pop up (>="1")
     **/
    public void setPop(String levelDesc)
    {
        require_(levelDesc!=null,"setPop- nonzro level descriptor");
        m_popLevelDesc = levelDesc;
    }
 
 
 
    /**
     * Gives this task a subpath to append to the calculated parent
     * directory. This subpath is appended after the parent has been
     * verified; any leading slashes are stripped.
     * @param subpath sub path (non-null)
     **/
    public void setAppend(String subpath)
    {
        require_(subpath!=null,"setAppend- nonzro subpath");
        final int N= subpath.length();
        int i=0;
        while (i<N) {
            if (subpath.charAt(0)==File.separatorChar) {
                subpath= subpath.substring(1);
            }
            i++;
        }
        m_subpath = subpath;
    }




    /**
     * Returns the subpath that this task will append to the calculated
     * parent directory. Will return <i>null</i> if never set.
     **/
    public final String getAppendedSubpath()
    {
        return m_subpath;
    }



    /**
     * Ensures we have at least one update property and a source file
     * reference. If no file reference was named, we assume we should
     * check our project's <span class="src">basedir</span> property.
     * @param calr calling function (non-null)
     * @throws BuildException if missing required update property.
     */
    protected void verifyCanExecute_(String calr)
    {
        verifyInProject_(calr);

        if (getPathProperty()==null && getUrlPathProperty()==null) {
            String e = getAntXMsg("task.needs.this.attr", getTaskName(),
                "pathproperty|urlproperty");
            log(e, Project.MSG_ERR);
            throw new BuildException(e,getLocation());
        }
        
        if (getOf()==null) {
            m_fileref = getProject().getProperty("basedir");
            verify_(m_fileref!=null,calr+"- Ant's basedir defined");
        }
    }



    private File theParent(File f)
    {
        File pf = f;
        int i = Tk.integerFrom(m_popLevelDesc,-1);
        while (pf!=null && i>0) {
            pf = pf.getParentFile();
            i--;
        }
        return pf;
    }



    /**
     * Tries to determine the parent directory of named file object.
     * If the object has no parent directory (is root), the update
     * property is set to the platform path string for root.
     * @throws BuildException if missing required update property.
     * @.impl Lots of calls to Project.resolvePath are necessary to deal
     *        with the mixing of platform dependent and unix-only path
     *        strings. The final path must be normalized to runtime's OS.
     */
    public void execute()
    {
        verifyCanExecute_("exec");

        File f= null;
        String filepath= getOf();

        if (filepath.startsWith("file:")) {
            filepath = AntXFixture.fileUtils().fromURI(filepath);
            f = getProject().resolveFile(filepath);
        }
        else if (filepath.startsWith("resource:")) {
            f = LocalTk.getSystemResourceSource(filepath.substring(9),getProject());
        }
        else if (filepath.startsWith("tmp:")) {
            File tmp = TempLocator.getSystemTempDir();
            filepath = filepath.substring(4);
            if (filepath.startsWith("/")) {
                filepath = filepath.substring(1);
            }
            filepath = tmp.getPath() + File.separator + filepath;
            f = getProject().resolveFile(filepath);
        } else {
            f = getProject().resolveFile(filepath);
        }

        File pf = theParent(f);
        if (getMustExist() && (pf==null || !pf.canRead())) {
            String e = getAntXMsg("mktemp.parent.notfound",m_popLevelDesc,getOf());
            log(e, Project.MSG_ERR);
            throw new BuildException(e,getLocation());
        }

        String parentPath = "";
        if (pf!=null) {
            parentPath = pf.getPath();
        } else if (f!=null) {
            //Find whatever OS seez is "root" path!
            parentPath = getProject().resolveFile("/").getPath();
        }
        if (!Tk.isWhitespace(m_subpath)) {
            parentPath += File.separator + m_subpath;
            parentPath = getProject().resolveFile(parentPath).getPath();
        }
        saveFinalPath(parentPath,true);
    }



    /**
     * Apply any common update properties to final file system object.
     * @param finalPath file system object's reference (non-null)
     * @param strict <i>true</i> if should fail if unable to set update properties
     **/
    private void saveFinalPath(String finalPath, boolean strict)
    {
        if (getPathProperty()!=null) {
            checkIfProperty_(getPathProperty(),!strict);
            getProject().setNewProperty(getPathProperty(),finalPath);
        }
        if (getUrlPathProperty()!=null) {
            checkIfProperty_(getUrlPathProperty(),!strict);
            String urlPath = AntXFixture.fileUtils().toURI(finalPath);
            getProject().setNewProperty(getUrlPathProperty(),urlPath);
        }
    }


    private String m_fileref;
    private String m_updateProperty;
    private String m_updateUrlProperty;
    private boolean m_mustExist;
    private String m_popLevelDesc= "1";//=> .. => parent
    private String m_subpath;//=> none
}


/* end-of-ParentDirTask.java */