/**
 * $Id: InstanceFactory.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.lang.reflect.Constructor;
import  java.util.Hashtable;

import  org.apache.tools.ant.BuildException;

/**
 * Utility class that acts as a simple name to implementation class mapping and
 * factory all in one. Like a standard <span class="src">Hashtable</span>, instances
 * of this class are protected against concurrent access and/or modification.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   guarded
 * @.group    impl,helper
 * @.pattern  GoF.Factory
 **/

public final class InstanceFactory extends Hashtable
{
    /** Signature of general parameter-based constructor. **/
    public static final Class[] CTOR_SIG0= new Class[] {GenericParameters.class};



    /**
     * Initializea new empty instance factory.
     **/
    public InstanceFactory()
    {
    }



    /**
     * Initializea new empty instance factory that uses a
     * specific default class for unmatched categories.
     * @param defaultClass the default class reference (non-null)
     **/
    public InstanceFactory(Class defaultClass)
    {
       setDefaultInstanceClass(defaultClass);
    }




    /**
     * Initializes the fallback class of unmatched categories.
     * @param c the class reference (non-null)
     * @throws IllegalArgumentException is class is <i>null</i>
     **/
    public void setDefaultInstanceClass(Class c)
    {
        if (c==null) {
            throw new IllegalArgumentException();
        }
        m_defaultClass = c;
    }



    /**
     * Returns this factory's fallback class for unmatched
     * categories. Will never return <i>null</i>.
     **/
    public Class getDefaultInstanceClass()
    {
        return m_defaultClass;
    }



    /**
     * Installs a new class mapping under a specific name.
     * @param key the category name (non-null)
     * @param value the <span class="src">Class</span> reference
     * @throws IllegalArgumentException if value is not a valid class reference.
     **/
    public Object put(Object key, Object value)
    {
        if (value==null) {
            return super.remove(key);
        }
        if (!(value instanceof Class)) {
            throw new IllegalArgumentException("Only class references allowed!");
        }
        return super.put(key,value);
    }



    private Class getc(Object key)
    {
        Object rc = get(key);
        Class c;
        if (rc==null) {
            c = getDefaultInstanceClass();
        } else {
            c = (Class)rc;
        }
        return c;
    }



    /**
     * Creates a new instance of the named category.
     * @param key the category's name (non-null)
     * @return new instance
     * @throws BuildException if unable to create new instance.
     **/
    public Object newInstance(Object key)
    {
        Class c = getc(key);
        try {
            return c.newInstance();
        } catch(Exception anyX) {
            throw new BuildException(anyX);
        }
    }



    /**
     * Creates a new custom instance of the named category.
     * @param key the category's name (non-null)
     * @param args the constructor's single argument (non-null)
     * @return new cstom instance
     * @throws BuildException if unable to create new instance.
     **/
    public Object newInstance(Object key, GenericParameters args)
    {
        Class c = getc(key);
        try {
            Constructor ctor = c.getConstructor(CTOR_SIG0);
            Object[] initargs = new Object[] {args};
            return ctor.newInstance(initargs);
        } catch(Exception anyX) {
            throw new BuildException(anyX);
        }
    }


    private Class m_defaultClass = Object.class;
}


/* end-of-InstanceFactory.java */