/**
 * $Id: GenericParameters.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.helpers;

import  java.util.Collection;
import  java.util.Iterator;
import  java.util.Map;
import  java.util.Properties;
import  java.util.StringTokenizer;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.ProjectComponent;

import  com.idaremedia.apis.FixtureStarter;

/**
 * Simple collection of {@linkplain InnerNameValuePair key-value} pairs. Duplicates are
 * permitted (last item added wins). Every parameter's name is normalized to its
 * US lowercase equivalent to ensure they can be used directly with Ant's macro and
 * dynamic attribute facilities. GenericParameters can be used as internal data structs,
 * or public-facing script components.
 * <p/>
 * <b>Example Usage:</b><pre>
 *   &lt;parameters&gt;
 *      &lt;parameter name="name" value="Larry"/&gt;
 *      &lt;parameter name="weight" value="165lbs"/&gt;
 *   &lt;/parameters&gt;
 * 
 *   -OR (shorthand)-
 *
 *   &lt;parameters list="prune;tag=latest"/&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 **/

public class GenericParameters extends ProjectComponent implements Cloneable
{
    /**
     * Initializes a new empty parameters collection bean.
     **/
    public GenericParameters()
    {
    }



    /**
     * Initializes a new empty parameters collection bean
     * associated with project.
     * @param project this parameter object's enclosing project
     **/
    public GenericParameters(Project project)
    {
        setProject(project);
    }



    /**
     * Returns an independent (deep) clone of this parameters
     * set. The individual items of the clone are also independent;
     * changes to them are not reflected in this set.
     */
    public Object clone()
    {
        try {
           GenericParameters copy= (GenericParameters)super.clone();
           copy.m_items = FixtureStarter.newMapCopy(m_items);
           if (size()>0) {
               Iterator itr= copy.m_items.entrySet().iterator();
               while (itr.hasNext()) {
                   Map.Entry e = (Map.Entry)itr.next();
                   e.setValue(((InnerNameValuePair)e.getValue()).clone());
               }
           }
           return copy;
        } catch(CloneNotSupportedException clnx) {
            throw new Error();
        }
    }



    /**
     * Adds a new name-value pair to this collection. The new
     * item is <em>not</em> checked for duplicity. If another item
     * with equivalent name already exists, it is replaced
     * unconditionally.
     * @param item new configured item (must be named)
     * @throws BuildException if incoming item not named.
     **/
    public void addConfiguredParameter(InnerNameValuePair item)
    {
        item.verifyNamed();
        item.setName(Tk.lowercaseFrom(item.getName()));
        m_items.put(item.getName(),item);
    }



    /**
     * Synonym for {@linkplain #addConfiguredParameter addConfiguredParameter}
     * that is more appropriate for some uses of this bean.
     * @param item new configured item (must be named)
     */
    public final void addConfiguredArg(InnerNameValuePair item)
    {
        addConfiguredParameter(item);
    }



    /**
     * Synonym for {@linkplain #addConfiguredParameter addConfiguredParameter}
     * that is more appropriate for some uses of this bean.
     * @param item new configured item (must be named)
     */
    public final void addConfiguredProperty(InnerNameValuePair item)
    {
        addConfiguredParameter(item);
    }



    /**
     * Adds a collection of name-value pairs to this collection 
     * using a simple semi-colon delimited list. If the list is
     * <i>null</i>, this collection is cleared of all current
     * entries. Each item in the list be in one of the following
     * forms:<ul>
     *   <li>&lt;name&gt;: where the value is automatically assigned as "true".</li>
     *   <li>&lt;name=&gt; where the value is assigned as the empty string.</li>
     *   <li>&lt;name=value&gt; where the value is assigned whatever "value" is.</li>
     * </ul>
     * @param paramlist the pairs
     **/
    public void setList(String paramlist)
    {
        if (paramlist!=null) {
            StringTokenizer st= new StringTokenizer(paramlist,";");
            while (st.hasMoreTokens()) {
                
                String kv = st.nextToken();
                InnerNameValuePair nvp = newNVPair();
                int i= kv.indexOf("=");
                
                if (i>0) {
                    String key = kv.substring(0,i);
                    String value = "";
                    if (i<kv.length()-1) {
                        value = kv.substring(i+1);
                    }
                    nvp.setName(key);
                    nvp.setValue(value);
                } else {
                    nvp.setName(kv);
                    nvp.setValue(Strings.TRUE);
                }
                
                addConfiguredParameter(nvp);
            }
        } else {
            m_items.clear();
        }
    }



    /**
     * Returns <i>true</i> if this collection is empty.
     **/
    public final boolean isEmpty()
    {
        return m_items.isEmpty();
    }



    /**
     * Returns the number of parameters in this collection.
     **/
    public final int size()
    {
        return m_items.size();
    }



    /**
     * Returns this parameters object's underlying name-value
     * items collection. The returned collection is still connected
     * to this data object, so modifications via its iterators are
     * reflected back to this object. Never returns <i>null</i>.
     **/
    public final Collection values()
    {
        return m_items.values();
    }



    /**
     * Returns an independent map of this collection's
     * name-value items. The returned map contains 
     * {@linkplain InnerNameValuePair} items as values.
     **/
    public Map copyOfParameterObjects()
    {
        return FixtureStarter.newMapCopy(m_items);
    }



    /**
     * Returns an independent Properties map of this collection's
     * name-value items as simplified name-stringvalue pairs.
     * The returned map contains the item names and string
     * values. Property references are resolved from given project.
     * @param P [optional] project used to resolve property references.
     * @param altForm alternative form of property references allowed.
     **/
    public Properties copyOfSimpleKeyValues(final Project P, boolean altForm)
    {
        Properties copy = new Properties();
        if (!m_items.isEmpty()) {
            Iterator itr= m_items.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry e = (Map.Entry)itr.next();
                copy.put(e.getKey(),
                    ((InnerNameValuePair)e.getValue()).getValue(P,altForm));
            }
        }
        return copy;
    }



    /**
     * Returns an independent Properties map of this collection's
     * name-value items as simplified name-stringvalue pairs.
     * The returned map contains the item names and string
     * values.
     * @param P [optional] project used to resolve property references.
     **/
    public final Properties copyOfSimpleKeyValues(final Project P)
    {
        return copyOfSimpleKeyValues(P,false);
    }



    /**
     * Combine another parameters collection with this one.
     * Shorthand for calling {@linkplain #addConfiguredParameter
     * addConfiguredParameter} repeatedly. Existing parameters
     * are replaced unconditionally with newer versions.
     * @param other parameters to be merged (non-null)
     **/
    public void mergeOther(GenericParameters other)
    {
        Iterator itr = other.copyOfParameterObjects().values().iterator();
        while (itr.hasNext()) {
            addConfiguredParameter((InnerNameValuePair)itr.next());
        }
    }



    /**
     * Combine another parameters collection in form of a standard
     * Java Properties object. Existing parameters are replaced
     * unconditionally with newer versions.
     * @param other other name-value pairs to be merged (non-null)
     **/
    public void mergeOther(Properties other)
    {
        Iterator itr = other.entrySet().iterator();
        while (itr.hasNext()) {
            Map.Entry e = (Map.Entry)itr.next();
            put(e.getKey().toString(),(String)e.getValue());
        }
    }



    /**
     * Convenient look method for a particular parameter.
     * @param name parameter's name (non-null)
     * @return parameter or <i>null</i> if no match
     **/
    public InnerNameValuePair get(String name)
    {
        name = Tk.lowercaseFrom(name);
        return (InnerNameValuePair)m_items.get(name);
    }



    /**
     * Convenient look method for a particular parameter.
     * @param name parameter's name (non-null)
     * @return parameter or <i>null</i> if no match
     **/
    public final String getv(String name)
    {
        InnerNameValuePair e = get(name);
        if (e!=null) {
            return e.getValue();
        }
        return null;
    }



    /**
     * Factory method to create a new name-value pair for inclusion
     * in this collection. By default returns standard InnerNameValuePair
     * objects.
     * @since JWare/AntX 0.5
     **/
    protected InnerNameValuePair newNVPair()
    {
        return new InnerNameValuePair();
    }



    /**
     * Common function to put a single name-value pair into this
     * parameters object.
     * @param name new item's unnormalized name (non-null)
     * @param value new item's value
     **/
    protected void put(String name, String value)
    {
        InnerNameValuePair item = newNVPair();
        item.setName(Tk.lowercaseFrom(name.toString()));
        item.setValue(value);
        m_items.put(item.getName(),item);
    }

    private Map m_items = FixtureStarter.newMap(11,0.8f);
}

/* end-of-GenericParameters.java */