/**
 * $Id: StepLauncher.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.flowcontrol.call;

import  java.util.ArrayList;
import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableTask;
import  com.idaremedia.antx.flowcontrol.FlowConstants;
import  com.idaremedia.antx.ownhelpers.TaskExaminer;

/**
 * Special task that receives a calling target and step name from an source parent
 * Ant process.  The launcher task will then try to execute the tasks nested within
 * a local instance of the the calling target and step.
 * <p>
 * The entire 'Ant' and 'CallTarget' mechanism is based on reloading/reparsing an
 * existing build file, so it is incapable of handling dynamically added targets
 * (which better match the concept of a step, but alas the default ProjectHelper
 * requires a real file-system build file to read/parse). This launcher task must
 * be nested inside a special target the originating StepCallers will use.
 * <p>
 * When executed launchers execute the nested tasks in a single step, much like
 * the builtin '<i>sequential</i>' task.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 * @see      InlinedTarget
 * @see      StepLauncherCaller
 * @see      StepLauncherInstallTask
 **/

public class StepLauncher extends AssertableTask
{
    /**
     * Creates a new step launcher task.
     **/
    public StepLauncher()
    {
        super(AntX.flow+"steplauncher");
    }


    /**
     * Optionally delays this launcher's configuration if I'm being
     * swapped with a placeholder.
     * @see TaskExaminer#ignoreConfigure TaskExaminer.ignoreConfigure(&#8230;)
     * @since JWare/AntX 0.4
     **/
    public void maybeConfigure()
        throws BuildException
    {
        if (!TaskExaminer.ignoreConfigure(this)) {
            super.maybeConfigure();
        }
    }



    /**
     * Helper to broadcast missing information to log/runtime.
     **/
    private BuildException missingInformation(String msgId, String arg1)
    {
        String ermsg = uistrs().get(msgId,arg1);
        log(ermsg,Project.MSG_ERR);
        return new BuildException(ermsg, getLocation());
    }


    /**
     * Checks all required pre-conditions for execute method. By default
     * ensures in valid project and all required properties are defined.
     * @throws BuildException if any precondition not met
     **/
    protected void verifyCanExecute_(String calr)
        throws BuildException
    {
        verifyInProject_(calr);

        String targetProperty = getTargetProperty();
        require_(targetProperty!=null,"exec- nonzro target property");

        String nameProperty = getInlinedTargetNameProperty();
        require_(nameProperty!=null,"exec- nonzro stepname property");

        final Project P= getProject();

        if (P.getProperty(targetProperty)==null) {
            throw missingInformation("flow.steplaunch.need.target",targetProperty);
        }
        if (P.getProperty(nameProperty)==null) {
            throw missingInformation("flow.steplaunch.need.stepname",nameProperty);
        }
    }



    /**
     * Returns the local target that matches the owning target of the
     * step-caller that triggered this launcher. By default this launcher
     * assumes the calling task inserted its target's name and the required
     * step name into two specially named properties.
     * @see #getTargetProperty
     * @throws BuildException if unable to find local target match
     **/
    protected Target getLocalTarget()
        throws BuildException
    {
        Target localTarget = (Target)getProject().getTargets().get(getTargetName());

        if (localTarget==null) {
            throw missingInformation("flow.steplaunch.missing.caller",
                                     getTargetName());
        }
        return localTarget;
    }


    /**
     * Returns a kind-of subset of tasks from a set of generic tasks. All
     * returned items implement the filter class or interface.
     * @param tasks the set of tasks to filter and update (non-null)
     * @param ofKind the filter class or interface (non-null)
     **/
    protected final List getTasksOfKind(Task[] tasks, Class ofKind)
        throws BuildException
    {
        List l= new ArrayList(tasks.length);
        Class[] COI= new Class[]{ofKind};
        for (int i=0;i<tasks.length;i++) {
            Task task = TaskExaminer.trueTask(tasks[i],COI,this);
            if (ofKind.isAssignableFrom(task.getClass())) {
                l.add(task);
                if (task!=tasks[i]) {//@since AntX0.4+Ant1.6.1
                    tasks[i]= task;  //replace Unknown with Known!
                }
            }
        }
        return l;
    }


    /**
     * Extracts a named inlined target from within a set of generic tasks. Iff
     * the inline-target is not found within the given top-level task set, nested
     * inline-target containers are searched, in order, up to '<i>depthLimit</i>'
     * deep <em>each</em>. Returns <i>null</i> if no matching inlined-target found.
     * @param stepName the inlined-target's name (non-whitespace)
     * @param filterClass [optional] if non-null the matching target must be of this class
     * @param tasks the set of possible tasks (non-null)
     * @param depth the current search depth
     * @param depthLimit the maximum search depth
     **/
    protected InlinedTarget getInlinedTarget(final String stepName, final Class filterClass,
                                             Task[] tasks, int depth, final int depthLimit)
    {
        InlinedTarget it = null;
        List candidates = getTasksOfKind(tasks, InlinedTarget.class);

        if (!candidates.isEmpty()) {
            Iterator itr= candidates.iterator();
            while (itr.hasNext()) {
                it = (InlinedTarget)itr.next();
                //NB: handles delayed maybeConfigure (pre Ant 1.6)
                if (it.getName()==null && (it instanceof Task)) {
                    ((Task)it).maybeConfigure();
                }
                if (stepName.equals(it.getName())) {
                    if (filterClass!=null && !filterClass.isInstance(it)) {
                        continue;
                    }
                    return it;
                }
            }
            it = null;
            candidates.clear();
        }
        candidates=null;

        if (depth<depthLimit) {
            candidates = getTasksOfKind(tasks, InlinedTargetContainer.class);
            if (!candidates.isEmpty()) {
                Iterator itr = candidates.iterator();
                Task[] nestedtasks;
                while (itr.hasNext()) {
                    InlinedTargetContainer itc = (InlinedTargetContainer)itr.next();
                    nestedtasks = itc.getTasks();
                    if (nestedtasks.length>0) {
                        it = getInlinedTarget(stepName, filterClass, nestedtasks,
                                              depth+1,depthLimit);
                        if (it!=null) {
                            break;
                        }
                    }
                }
                candidates.clear();
            }
            candidates=null;
        }

        return it;
    }


    /**
     * Returns the local instance of the requested step within the local target.
     * Returns <i>null</i> if no local step matches criteria.
     * @param stepName caller's inlined-targets name (non-null)
     * @param requiredClass [optional] further refinement of the acceptable target
     * @throws BuildException if unable to load requiredClass reference or unable
     *         to locate local target.
     **/
    protected InlinedTarget findInlinedTarget(String stepName, String requiredClass)
        throws BuildException
    {
        require_(stepName!=null,"fndInlinTargt- nonzro name");

        Class filterClass = null;
        if (requiredClass!=null) {
            try {
                filterClass = Class.forName(requiredClass);
            } catch(ClassNotFoundException cnfx) {
                throw new BuildException(cnfx, getLocation());
            }
        }

        return getInlinedTarget(stepName, filterClass,
                                getLocalTarget().getTasks(),
                                1, getMaxSearchDepth());
    }



    /**
     * Executes locally the inlined-target that matches the calling
     * step-caller's target. Note the calling step-caller should belong
     * to a separate project (and environment space).
     * @see #getInlinedTargetNameProperty
     * @see #findInlinedTarget
     * @throws BuildException if unable to locate local inlined-target or
     *         the inlined-target throws an exception
     **/
    public void execute() throws BuildException
    {
        verifyCanExecute_("execute");

        InlinedTarget it = findInlinedTarget(getInlinedTargetName(),
                                             getKindOfInlinedTargetClass());
        if (it==null) {
            String ermsg = uistrs().get("flow.steplaunch.missing.step",
                                        getInlinedTargetName(), getTargetName());
            log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg, getLocation());
        }

        it.run();
    }



    /**
     * Returns the property that contains the local target's name. Must
     * be defined from the step caller's "call-target" code.
     * @see FlowConstants#DEFAULT_STEP_LAUNCHER_TARGET_PROP
     **/
    protected String getTargetProperty()
    {
        return FlowConstants.DEFAULT_STEP_LAUNCHER_TARGET_PROP;
    }


    /**
     * Returns the property that contains the local inlined-target's name.
     * Must be defined from the step-caller's "call-target" code.
     * @see FlowConstants#DEFAULT_STEP_LAUNCHER_STEPNAME_PROP
     **/
    protected String getInlinedTargetNameProperty()
    {
        return FlowConstants.DEFAULT_STEP_LAUNCHER_STEPNAME_PROP;
    }



    /**
     * Returns class name of custom {@linkplain InlinedTarget InlinedTarget}
     * subclass that is being targeted with this launcher. By default assumes only
     * {@linkplain InlineStep InlineStep} implementations are targeted.
     * @see FlowConstants#DEFAULT_STEP_LAUNCHER_STEPCLASS_PROP
     **/
    protected String getKindOfInlinedTargetClass()
    {
        String kindOfInlinedTargetClass =
            getProject().getProperty
                (FlowConstants.DEFAULT_STEP_LAUNCHER_STEPCLASS_PROP);

        if (kindOfInlinedTargetClass==null) {
            kindOfInlinedTargetClass = InlineStep.class.getName();
        }

        return kindOfInlinedTargetClass;
    }



    /**
     * Returns calling target's name.
     **/
    private String getTargetName()
    {
        return getProject().getProperty(getTargetProperty());
    }


    /**
     * Returns called inlined-target's name.
     **/
    private String getInlinedTargetName()
    {
        return getProject().getProperty(getInlinedTargetNameProperty());
    }


    /**
     * Returns maximum search depth for inlined-targets. Returned number
     * must be equal-to or greater than one (1).
     **/
    protected int getMaxSearchDepth()
    {
        String s =
            getProject().getProperty
                (FlowConstants.DEFAULT_STEP_LAUNCHER_SEARCHDEPTH_PROP);
        if (s!=null) {
            try {
                int i = Integer.parseInt(s);
                if (i>0) {
                    return i;
                }
            } catch(NumberFormatException nfx) {/*burp*/}
            String ermsg = uistrs().get("flow.steplaunch.bad.depth",s,"1");
            log(ermsg,Project.MSG_WARN);
        }
        return 1;
    }

}

/* end-of-StepLauncher.java */
