/**
 * $Id: MainField.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.EnumSkeleton;

/**
 * A BuildEvent's main information categories. MainFields are used to create a default
 * (minimal) {@linkplain EventEmitConduit hierarchy} of captured Ant log messages. They
 * also form the basis for user-controlled category {@linkplain GroupingMapper groupings}.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   impl,helper
 * @see      EventEmitConduit
 * @see      GroupingMapper
 **/

public final class MainField extends EnumSkeleton
{
    public static final int INCLUDE_THREAD  = 0x01;
    public static final int INCLUDE_PROJECT = 0x02;
    public static final int INCLUDE_TARGET  = 0x04;
    public static final int INCLUDE_TASK    = 0x08;
    public static final int INCLUDE_NESTED  = 0x10;
    public static final int INCLUDE_ALL     = 0xff;
    public static final int DEFAULT_FILTER  = INCLUDE_ALL;


    /** Index of {@linkplain #PROJECT PROJECT}. **/
    public static final int PROJECT_INDEX= 0;
    /** Index of {@linkplain #TARGET TARGET}. **/
    public static final int TARGET_INDEX= PROJECT_INDEX+1;
    /** Index of {@linkplain #TASK TASK}. **/
    public static final int TASK_INDEX= TARGET_INDEX+1;
    /** Index of {@linkplain #MESSAGE MESSAGE}. **/
    public static final int MESSAGE_INDEX= TASK_INDEX+1;
    /** Index of {@linkplain #INDICATOR INDICATOR}. **/
    public static final int INDICATOR_INDEX= MESSAGE_INDEX+1;
    /** Index of {@linkplain #UNKNOWN UNKNOWN}. **/
    public static final int UNKNOWN_INDEX= INDICATOR_INDEX+1;

    /** Singleton "project" field. **/
    public static final MainField PROJECT =
        new MainField("project",PROJECT_INDEX,INCLUDE_PROJECT);
    /** Singleton "target" field. **/
    public static final MainField TARGET =
        new MainField("target",TARGET_INDEX,INCLUDE_TARGET);
    /** Singleton "task" field. **/
    public static final MainField TASK =
        new MainField("task",TASK_INDEX,INCLUDE_TASK);
    /** Singleton "message" field. **/
    public static final MainField MESSAGE =
        new MainField("message",MESSAGE_INDEX, INCLUDE_ALL);
    /** Singleton "indicator" field. **/
    public static final MainField INDICATOR =
        new MainField("indicator",INDICATOR_INDEX, INCLUDE_ALL);
    /** Singleton unknown field. **/
    public static final MainField UNKNOWN =
        new MainField("_any_",UNKNOWN_INDEX, INCLUDE_ALL);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public MainField()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensure it's initialized
     * as with default Ant Introspector helper thingy.
     **/
    private MainField(String v, int i, int filter)
    {
        super(v);
        m_filter = filter;
    }


    /**
     * Returns copy of all possible mainfield values as an ordered
     * string array. Note: ordering should be same as singletons indice.
     **/
    public String[] getValues()
    {
        return new String[] {"project","target","task","message",
                             "indicator","_any_"};
    };


    /**
     * Returns this fields associated filter or zero (0) if none.
     **/
    final int getFilter()
    {
        return m_filter;
    }


    /**
     * Helper that converts a scalar to a known MainField. Returns
     * <i>null</i> if value does not match any of expected fields.
     * @param i the index to be matched
     **/
    public static MainField from(int i)
    {
        if (i==TASK.index)      { return TASK; }
        if (i==MESSAGE.index)   { return MESSAGE; }
        if (i==TARGET.index)    { return TARGET; }
        if (i==PROJECT.index)   { return PROJECT; }
        if (i==INDICATOR.index) { return INDICATOR; }
        if (i==UNKNOWN.index)   { return UNKNOWN; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known MainField's
     * index.
     * @param i the index to be matched
     * @param dflt the default MainField if necessary
     **/
    public static MainField from(int i, MainField dflt)
    {
        MainField mf= from(i);
        return (mf==null) ? dflt : mf;
    }


    /**
     * Helper that converts a string to a known MainField singleton.
     * Returns <i>null</i> if string unrecognized. String can be
     * either MainField's symbolic name or its index.
     **/
    public static MainField from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (TASK.value.equals(s))      { return TASK; }
                if (MESSAGE.value.equals(s))   { return MESSAGE; }
                if (TARGET.value.equals(s))    { return TARGET; }
                if (PROJECT.value.equals(s))   { return PROJECT; }
                if (INDICATOR.value.equals(s)) { return INDICATOR; }
                if (UNKNOWN.value.equals(s))   { return UNKNOWN; }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(java.lang.String) from(String)} but
     * with a default value if value does not match any known
     * MainField's name.
     * @param s the symbolic name to be matched
     * @param dflt the default MainField if necessary
     **/
    public static MainField from(String s, MainField dflt)
    {
        MainField mf= from(s);
        return (mf==null) ? dflt : mf;
    }


    private int m_filter;
}

/* end-of-MainField.java */
