/**
 * $Id: IndicatorZone.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.EnumSkeleton;

/**
 * Emitted feedback's possible indicator zones. Three options: <i>Problems</i>,
 * <i>Status</i>, and <i>Diagnostics</i>.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple
 * @.group   api,infra
 **/

public final class IndicatorZone extends EnumSkeleton
{
    /** Index of {@linkplain #PROBLEMS PROBLEM}. **/
    public static final int PROBLEMS_INDEX= 0;
    /** Index of {@linkplain #STATUS STATUS}. **/
    public static final int STATUS_INDEX= PROBLEMS_INDEX+1;
    /** Index of {@linkplain #DIAGNOSTICS DIAGNOSTICS}. **/
    public static final int DIAGNOSTICS_INDEX= STATUS_INDEX+1;

    /** Singleton "problems" indicator zone. **/
    public static final IndicatorZone PROBLEMS =
        new IndicatorZone("problems",PROBLEMS_INDEX);
    /** Singleton "status" indicator zone. **/
    public static final IndicatorZone STATUS =
        new IndicatorZone("status",STATUS_INDEX);
    /** Singleton "diagnostics" indicator zone. **/
    public static final IndicatorZone DIAGNOSTICS =
        new IndicatorZone("diagnostics",DIAGNOSTICS_INDEX);


    /**
     * Required bean void constructor for Ant's introspector.
     **/
    public IndicatorZone()
    {
        super();
    }


    /**
     * Use to create public singletons. Ensure it's initialized
     * as with default Ant Introspector helper thingy.
     **/
    private IndicatorZone(String v, int i)
    {
        super(v);
    }


    /**
     * Returns copy of all possible indicator-zone values as an ordered
     * string array. Note: ordering should be same as singletons indice.
     **/
    public String[] getValues()
    {
        return new String[] {"problems","status","diagnostics"};
    };



    /**
     * Helper that converts a scalar to a known IndicatorZone. Returns
     * <i>null</i> if value does not match any of expected indicator-zones.
     * @param i the index to be matched
     **/
    public static IndicatorZone from(int i)
    {
        if (i==DIAGNOSTICS.index) { return DIAGNOSTICS; }
        if (i==STATUS.index)      { return STATUS; }
        if (i==PROBLEMS.index)    { return PROBLEMS; }
        return null;
    }


    /**
     * Same as {@linkplain #from(int) from(int)} but with a
     * default value if value does not match any known IndicatorZone's
     * index.
     * @param i the index to be matched
     * @param dflt the default IndicatorZone if necessary
     **/
    public static IndicatorZone from(int i, IndicatorZone dflt)
    {
        IndicatorZone iz= from(i);
        return (iz==null) ? dflt : iz;
    }


    /**
     * Helper that converts a NoiseLevel to a known IndicatorZone
     * singleton. Returns <i>null</i> if noise level is <i>null</i>.
     * @param nl noiselevel
     **/
    public static IndicatorZone from(NoiseLevel nl)
    {
        if (nl!=null) {
            switch(nl.getIndex()) {
                case NoiseLevel.FATAL_INDEX:
                case NoiseLevel.ERROR_INDEX:
                case NoiseLevel.WARNING_INDEX: {
                    return PROBLEMS;
                }
                case NoiseLevel.INFO_INDEX: {
                    return STATUS;
                }
                default: {
                    return DIAGNOSTICS;
                }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(NoiseLevel) from(NoiseLevel)} but
     * with a default value if value does not match any known
     * IndicatorZone's noise levels.
     * @param nl the noise level to be matched
     * @param dflt the default IndicatorZone if necessary
     **/
    public static IndicatorZone from(NoiseLevel nl, IndicatorZone dflt)
    {
        IndicatorZone iz= from(nl);
        return (iz==null) ? dflt : iz;
    }


    /**
     * Helper that converts a string to a known IndicatorZone singleton.
     * Returns <i>null</i> if string unrecognized. String can be
     * either IndicatorZone's symbolic name, its index, or a noiselevel's
     * name.
     **/
    public static IndicatorZone from(String s)
    {
        if (s!=null && s.length()>1) {
            s = Tk.lowercaseFrom(s);
            if (Character.isDigit(s.charAt(0))) {
                try { return from(Integer.parseInt(s)); }
                catch(Exception nfx) {/*burp*/}
            } else {
                if (DIAGNOSTICS.value.equals(s)) { return DIAGNOSTICS; }
                if (STATUS.value.equals(s))      { return STATUS; }
                if (PROBLEMS.value.equals(s))    { return PROBLEMS; }

                NoiseLevel nl= NoiseLevel.from(s);
                if (nl!=null) {
                    return from(nl);
                }
            }
        }
        return null;
    }


    /**
     * Same as {@linkplain #from(String) from(String)} but
     * with a default value if value does not match any known
     * IndicatorZone's or NoiseLevel's name.
     * @param s the symbolic name to be matched
     * @param dflt the default IndicatorZone if necessary
     **/
    public static IndicatorZone from(String s, IndicatorZone dflt)
    {
        IndicatorZone iz= from(s);
        return (iz==null) ? dflt : iz;
    }
}

/* end-of-IndicatorZone.java */
