/**
 * $Id: GroupingMapper.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2003-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  java.util.Iterator;
import  java.util.List;

import  org.apache.tools.ant.BuildEvent;
import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Target;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.types.Reference;
import  org.apache.tools.ant.types.RegularExpression;
import  org.apache.tools.ant.util.RegexpPatternMapper;
import  org.apache.tools.ant.util.regexp.Regexp;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableDataType;
import  com.idaremedia.antx.FixtureComponent;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.IgnoreCaseEnabled;

/**
 * Type that maps a BuildEvent to a diagnostics grouping using the event's information
 * and a user-defined set of mappings. This class does <em>not</em> assume it's being
 * used solely by {@linkplain EventEmitConduit} items; it can handle "raw" build events
 * that haven't been massaged by a conduit. Needs to be called by a build listener
 * to be useful.
 * <p>
 * The primary task (client) methods are: {@linkplain #pathFrom pathFrom}, {@linkplain
 * #getLabel getLabel}, and {@linkplain #isEmpty isEmpty}. Most of the other methods are
 * used by Ant's introspection methods as they construct a mapper from a build script.
 * <p>
 * <b>Examples:</b><pre>
 *     &lt;emitmappings id="default.labels"&gt;
 *        &lt;mapping type="indicator" name="problems" label="OohTheHumanity"/&gt;
 *        &lt;mapping type="project" like=".*" label="AntX"/&gt;
 *     &lt;/emitmappings&gt;
 *
 *     &lt;emitmappings id="local.labels" inherit="default.labels"&gt;
 *        &lt;mapping type="task" name="property" label="Property"/&gt;
 *        &lt;mapping type="task" names="callforeach,foreach" label="ForEach"/&gt;
 *        &lt;mapping type="target" name="testABC" label="AABBCC"/&gt;
 *        &lt;mapping type="message"
 *               like="^AntX (task|type){1} \{(.*)\} loading resource .*"
 *               labelexpr="AntX\.Tools\.Resources\.\2"/&gt;
 *     &lt;/emitmappings&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  multiple (once fully configured)
 * @.group   impl,helper
 * @see      EmitLogsTask
 * @see      EventEmitConduit
 **/

public class GroupingMapper extends AssertableDataType
    implements FixtureComponent, Cloneable
{
// ---------------------------------------------------------------------------------------
// Individual Mappings
// ---------------------------------------------------------------------------------------
    /**
     * Mapping from a field match to a human-readable diagnostics category.
     * <br>
     * Some examples:<pre>
     *   &lt;mapping type="task" names="property,copyproperty" label="Property"/&gt;
     *   &lt;mapping type="target" name="--init" label="Initialization"/&gt;
     *   &lt;mapping type="target" like="^test.*" label="Tests"/&gt;
     * </pre>
     *
     * @since    JWare/AntX 0.3
     * @author   ssmc, &copy;2003-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  multiple (once fully configured)
     * @.group   impl,helper
     **/
    public static class Item extends AssertableDataType
        implements Cloneable, IgnoreCaseEnabled
    {
        private static final String ONLY_ONE_ERR= "task.only.oneof.attr";


        /**
         * Initializes a new partially-defined match item. This new
         * item's type and match criteria must be assigned before it
         * is used to test values.
         **/
        public Item()
        {
            super(AntX.feedback);
        }

        /**
         * Returns a full clone of this match item. The returned
         * copy includes an independent copy of all underlying criteria
         * elements (like regular expressions, etc.)
         **/
        public Object clone()
        {
            try {
                Item cloned = (Item)super.clone();
                if (m_re!=null) {
                    cloned.setLike(getLike());
                }
                return cloned;
            } catch(CloneNotSupportedException clnx) {
                throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
            }
        }

        /**
         * Captures a script-declared identifier for this match item.
         * @see #getId
         **/
        public void setId(String id)
        {
            m_Id= id;
        }

        /**
         * Returns this item's Ant reference identifier. Never returns
         * <i>null</i>.
         **/
        public String getId()
        {
            if (m_Id!=null) {
                return m_Id;
            }
            return super.getId();
        }

        /**
         * Initializes this item's source type category. Must be
         * done once before this item used for matching.
         * @see #getType
         **/
        public void setType(String f)
        {
            require_(f!=null,"setTyp- nonzro fld");
            m_field = MainField.from(f,MainField.UNKNOWN);//So we can get singletons
        }

        /**
         * Returns this item's associated source type category.
         * Returns the "<span class="isrc">UNKNOWN</span>" if never
         * assigned.
         **/
        public final MainField getType()
        {
            return m_field;
        }

        /**
         * Sets this item to match source identifier by name.
         * @param name the (exact) name that must match (non-null)
         * @throws BuildException if this item already assigned to match
         *         by any other means (like by-name)
         * @see #isIgnoreCase
         **/
        public void setName(String name)
        {
            require_(name!=null,"setNam- nonzro nam");
            if (m_re!=null || m_names!=null) {
                String error = uistrs().get(ONLY_ONE_ERR,"name|names|like");
                throw new BuildException(error);
            }
            m_name = name;
        }

        /**
         * Returns this item's match by-name value. Will return <i>null</i>
         * if this item isn't matching by name.
         **/
        public final String getName()
        {
            return m_name;
        }

        /**
         * Sets this item to match one of a set of names.
         * @param list the comma-delimited list of names (non-null,non-empty)
         * @throws BuildException if this item already assigned to match
         *         by any other means (like by-pattern)
         * @since JWare/AntX 0.4
         **/
        public void setNames(String list)
        {
            require_(list!=null,"setNamLst- nonzro liststr");
            if (m_re!=null || m_name!=null) {
                String error = uistrs().get(ONLY_ONE_ERR,"name|names|like");
                throw new BuildException(error);
            }
            List l= Tk.splitList(list);
            if (l.isEmpty()) {
                throw new BuildException(uistrs().get("emit.mapr.bad.namelist"));
            }
            m_names= (String[])l.toArray(new String[l.size()]);
            l=null;
        }

        /**
         * Returns a <em>copy</em> of the match-to name list for this
         * item. Will return <i>null</i> if this item isn't matching by
         * name list.
         * @since JWare/AntX 0.4
         **/
        public final String[] getNames()
        {
            String[] copy=null;
            if (m_names!=null) {
                copy= new String[m_names.length];
                for (int i=0;i<copy.length;i++) {
                    copy[i]= m_names[i];
                }
            }
            return copy;
        }

        /**
         * Sets this item to match source identifier by pattern. The
         * pattern must be compatible with the surrounding Ant runtime's
         * regular expression spi.
         * @param rexp the regular expression pattern (non-null)
         * @throws BuildException if this item already assigned to match
         *         by any other means (like by-name)
         **/
        public void setLike(String rexp)
        {
            require_(rexp!=null,"setRE- nonzro RE");
            if (m_name!=null || m_names!=null) {
                String error = uistrs().get(ONLY_ONE_ERR,"name|names|like");
                throw new BuildException(error);
            }
            m_re = new RegularExpression();
            m_re.setPattern(rexp);
        }

        /**
         * Returns this item's like expression as a pattern string.
         * Will return <i>null</i> if this item isn't matching by pattern.
         **/
        public final String getLike()
        {
            return (m_re!=null) ? m_re.getPattern(getProject()) : null;
        }

        /**
         * Returns this item's like expression as a full Regexp object.
         * Will return <i>null</i> if this item isn't matching by pattern.
         **/
        public final Regexp getLikeRE()
        {
            return (m_re!=null) ? m_re.getRegexp(getProject()) : null;
        }

        /**
         * Sets this item's script-assigned label. This label is used
         * to map into the external logging system's cataloging mechanism.
         * @param label the label (non-null)
         **/
        public void setLabel(String label)
        {
            require_(label!=null,"setLbl- nonzro lbl");
            if (m_isMapper!=null) {
                String error = uistrs().get(ONLY_ONE_ERR,"label|labelexpr");
                throw new BuildException(error);
            }
            m_label = label;
            m_isMapper = Boolean.FALSE;
        }

        /**
         * Sets this items's script-assigned label match regular expression.
         * This expression can contain grouping references of the
         * {@linkplain #setLike like} expression.
         * @since JWare/AntX 0.4
         **/
        public void setLabelExpr(String re)
        {
            require_(re!=null,"setLblExpr- nonzro re");
            if (m_isMapper!=null) {
                String error = uistrs().get(ONLY_ONE_ERR,"label|labelexpr");
                throw new BuildException(error);
            }
            m_label = re;
            m_isMapper = Boolean.TRUE;
        }


        /**
         * Returns this item's assigned label. Returns the empty string
         * if never assigned. Never returns <i>null</i>.
         **/
        public String getLabel()
        {
            return m_label;
        }

        /**
         * Returns target label if the given string is a match for this
         * item. Returns <i>null</i> if is not match.
         * @param s the string-under-test (can be <i>null</i>)
         **/
        public String isMatch(String s)
        {
            if (s==null) {
                return null;
            }
            if (m_isMapper!=null && m_isMapper.booleanValue()) {
                RegexpPatternMapper helper= new RegexpPatternMapper();
                helper.setTo(getLabel());
                helper.setFrom(getLike());
                String[] rslt = helper.mapFileName(s);
                helper = null;
                return (rslt!=null) ? rslt[0] : null;
            }

            boolean hit = false;;
            Regexp RE = getLikeRE();
            if (RE!=null) {
                hit= RE.matches(s,isIgnoreCase() ? Regexp.MATCH_CASE_INSENSITIVE: 0);
            }
            else if (m_names!=null) {
                boolean caseins= isIgnoreCase();
                for (int i=0;i<m_names.length && !hit;i++) {
                    if (caseins) {
                        if (m_names[i].equalsIgnoreCase(s)) {
                            hit = true;
                        }
                    } else {
                        if (m_names[i].equals(s)) {
                            hit = true;
                        }
                    }
                }
            }
            else {
                hit = isIgnoreCase() ? m_name.equalsIgnoreCase(s) : m_name.equals(s);
            }
            return hit ? getLabel() : null;
        }

        /**
         * Tells this match item to ignore case when matching values.
         * Applies to both names and patterns. Defaults <i>false</i>.
         * @param ignore <i>true</i> to ignore case
         **/
        public void setIgnoreCase(boolean ignore)
        {
            m_ignoreCase = ignore;
        }

        /**
         * Returns <i>true</i> if this match item will ignore case when
         * testing values.
         **/
        public boolean isIgnoreCase()
        {
            return m_ignoreCase;
        }

        /**
         * Unsupported inherited attribute.
         * @throws BuildException always
         **/
        public void setRefid(String xxx)
        {
            throw new BuildException
                (uistrs().get("task.unsupported.attr","mapping/item",
                              "refid"));
        }

        private String m_Id;
        private MainField m_field= MainField.UNKNOWN;
        private String m_label= "";
        private String m_name, m_names[];
        private RegularExpression m_re;
        private boolean m_ignoreCase;
        private Boolean m_isMapper;
    }

// ---------------------------------------------------------------------------------------
// Construction, Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Initializes a new empty GroupingMapper instance.
     **/
    public GroupingMapper()
    {
        super(AntX.feedback);
    }



    /**
     * Returns this mapping's reference strongly typed. This
     * mapper must be a reference.
     **/
    protected final GroupingMapper getOtherMapper()
    {
        return (GroupingMapper)getCheckedRef(GroupingMapper.class,"mappings");
    }



    /**
     * Returns a clone of this grouping mapper. Supports copying
     * of mappers between parent and forked child projects. If this
     * mapper is a reference, that referred-to object's clone method
     * is invoked.
     **/
    public Object clone()
    {
        if (isReference()) {
            return getOtherMapper().clone();
        }
        try {
            GroupingMapper cloned = (GroupingMapper)super.clone();
            cloned.m_items = AntXFixture.newListCopy(m_items);
            for (int i=0,N=cloned.m_items.size();i<N;i++) {
                cloned.m_items.set(i,((Item)m_items.get(i)).clone());
            }
            return cloned;
        } catch(CloneNotSupportedException clnx) {
            throw new Error(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }



    /**
     * Capture our identifier for feedback since types don't
     * always get correct location information.
     **/
    public void setId(String id)
    {
        m_Id= id;
    }



    /**
     * Returns a unique identifier for this GroupingMapper.
     **/
    public final String getId()
    {
        if (m_Id!=null) {
            return m_Id;
        }
        if (isReference()) {
            return getOtherMapper().getId();
        }
        return super.getId();
    }



    /**
     * Defines a referred-to mapper when there are no matches
     * in this mapper.
     * @param refid reference to existing mapper (non-null)
     * @since JWare/AntX 0.4
     **/
    public void setInherit(Reference refid)
    {
        require_(refid!=null,"setInherited- nonzro refid");
        m_referToMapper = refid;
    }



    /**
     * Returns this mapper's referred-to mapper for when there
     * are no local matches. Will return <i>null</i> if never
     * set explicitly.
     * @since JWare/AntX 0.4
     **/
    public synchronized final GroupingMapper getInheritedMapper()
    {
        if (m_referToMapper instanceof Reference) {
            m_referToMapper = getReferencedObject
                (null, ((Reference)m_referToMapper).getRefId(),
                 GroupingMapper.class);

            //FIXME: need proper circular-reference check here!
            verify_(m_referToMapper!=this,"inherit- not circular");
        }
        return (GroupingMapper)m_referToMapper;
    }



    /**
     * Adds a new user-supplied mapping to this mapper.
     * @param itm the new (fully-configured) mapping (non-null)
     * @throws BuildException if this mapper is a reference
     **/
    public void addConfiguredMapping(Item itm)
    {
        require_(itm!=null,"addMapin- nonzro itm");
        if (isReference()) {
            throw tooManyAttributes();
        }
        getItems().add(itm);
        edited();
    }



    /**
     * Returns <i>true</i> if this mapper contains no mappings.
     **/
    public final boolean isEmpty()
    {
        if (isReference()) {
            return getOtherMapper().isEmpty();
        }
        return m_items.isEmpty();
    }



    /**
     * Returns the underlying list of mapper items. Do not expose
     * via a public interface! Never returns <i>null</i>.
     **/
    protected final List getItems()
    {
        return m_items;
    }


// ---------------------------------------------------------------------------------------
// Matching event fields to custom label mappings:
// ---------------------------------------------------------------------------------------


    /**
     * Tries to determine a project's name. Never returns <i>null</i>.
     * @since JWare/AntX 0.4
     **/
    private static String projectName(Project project)
    {
        String pn = project.getName();
        if (pn==null) {
            pn = project.getProperty("ant.project.name");
            if (pn==null) {
                pn= "project@"+System.identityHashCode(project);
            }
        }
        return pn;
    }



    /**
     * Returns <i>true</i> if given item is a local match for a build event.
     * This method does <em>not</em> check this mapper's inherited mapper.
     * @param itm the item against which match is checked (non-null)
     * @param e the event being checked (non-null)
     * @param iz candidate zone for event (non-null)
     **/
    public boolean isMatch(Item itm, BuildEvent e, IndicatorZone iz)
    {
        require_(itm!=null && e!=null,"isMatch- nonzro event+item");
        require_(iz!=null,"isMatch- nonzro izone");

        if (isReference()) {
            return getOtherMapper().isMatch(itm,e,iz);
        }

        return isMatch_(itm,e,iz)!=null;
    }



    /**
     * Actual work method that determines if a local mapping matches a build
     * event. This is an internal method that assumes the caller has already
     * verified calling prerequisites.
     **/
    protected final String isMatch_(Item itm, BuildEvent e, IndicatorZone iz)
    {
        switch(itm.getType().getIndex()) {
            case MainField.TASK_INDEX: {
                if (e.getTask()!=null) {
                    return itm.isMatch(e.getTask().getTaskName());
                }
                break;
            }
            case MainField.TARGET_INDEX: {
                if (e.getTarget()!=null) {
                    return itm.isMatch(e.getTarget().getName());
                }
                break;
            }
            case MainField.PROJECT_INDEX: {
                if (e.getProject()!=null) {
                    return itm.isMatch(projectName(e.getProject()));
                }
                break;
            }
            case MainField.INDICATOR_INDEX: {
                return itm.isMatch(iz.getValue());
            }
            case MainField.MESSAGE_INDEX: {
                if (e.getMessage()!=null) {
                    return itm.isMatch(e.getMessage());
                }
                break;
            }
        }//switch
        return null;
    }

// ---------------------------------------------------------------------------------------
// Converting an event to a category grouping:
// ---------------------------------------------------------------------------------------

    /**
     * Returns best match label for a specific build event field. If
     * mapping not found locally or in inherited mapper, this method returns
     * <i>null</i>.
     * @param want [optional] the required field (non-null)
     * @param e the event being checked (non-null)(itm.getType()==want) {
     * @param iz candidate zone for event (non-null)
     **/
    public String getLabel(MainField want, BuildEvent e, IndicatorZone iz)
    {
        require_(want!=null && e!=null,"getLabel- nonzro event+field");
        require_(iz!=null,"isMatch- nonzro izone");

        if (isReference()) {
            return getOtherMapper().getLabel(want,e,iz);
        }
        return getLabel_(want,e,iz);
    }



    /**
     * Actual work method that determines a specific field's label. This
     * is an internal method that assumes the caller has already verified
     * calling prerequisites. Inherited mapper is also checked.
     **/
    protected final String getLabel_(MainField want, BuildEvent e, IndicatorZone iz)
    {
        if (!m_items.isEmpty()) {
            Iterator itr= m_items.iterator();
            while (itr.hasNext()) {
                Item itm = (Item)itr.next();
                if (itm.getType()==want) {
                    String label = isMatch_(itm, e, iz);
                    if (label!=null) {
                        return label;
                    }
                }
            }
        }
        if (getInheritedMapper()!=null) {
            return getInheritedMapper().getLabel_(want,e,iz);
        }
        return null;
    }



    /**
     * Determine the diagnostic grouping for the logged message event.
     * Never returns <i>null</i>. Does not match message mappings; caller
     * has that option. Will omit fields if incoming event has no
     * source information; for example, will omit task field if event was
     * triggered by a target's log event.
     * @param e the build event (non-null)
     * @param iz the root category zone (non-null)
     * @param allowed grouping hints (see MainField.INCLUDE_* constants)
     * @param fieldseparator string used to separate fields in returned path
     **/
    public String pathFrom(BuildEvent e, IndicatorZone iz,
                           int allowed, String fieldseparator)
    {
        require_(e!=null && iz!=null, "pathFrom- nonzro event+izone");

        if (isReference()) {
            return getOtherMapper().pathFrom(e,iz,allowed,fieldseparator);
        }

        final String SEP= (fieldseparator!=null) ? fieldseparator :
            DefaultEmitConfiguration.INSTANCE.getGroupingPathSeparator();

        StringBuffer sb = AntXFixture.newStringBuffer();
        String field;

        sb.append(indicatorLabel(e,iz,allowed));//NB:always

        if ((allowed&MainField.INCLUDE_PROJECT)!=0) {
            field = projectLabel(e,iz,allowed);
            if (field!=null) {
                sb.append(SEP);
                sb.append(field);
            }
        }
        if ((allowed&MainField.INCLUDE_TARGET)!=0) {
            field = targetLabel(e,iz,allowed);
            if (field!=null) {
                sb.append(SEP);
                sb.append(field);
            }
        }
        if ((allowed&MainField.INCLUDE_TASK)!=0) {
            field = taskLabel(e,iz,allowed);
            if (field!=null) {
                sb.append(SEP);
                sb.append(field);
            }
        }

        return sb.substring(0);
    }



    /**
     * Determine the diagnostic grouping for the logged message event.
     * Never returns <i>null</i>. Does not match message mappings; caller
     * has that option. Will insert placeholder field labels for fields
     * that event is missing.
     * @param e the build event (non-null)
     * @param iz the root category zone (non-null)
     * @param allowed grouping hints (see MainField.INCLUDE_* constants)
     * @param fieldseparator string used to separate fields in returned path
     * @since JWare/AntX 0.4
     **/
    public String pathFromAllFields(BuildEvent e, IndicatorZone iz,
                                    int allowed, String fieldseparator)
    {
        require_(e!=null && iz!=null, "pathFrom- nonzro event+izone");

        if (isReference()) {
            return getOtherMapper().pathFrom(e,iz,allowed,fieldseparator);
        }

        final String SEP= (fieldseparator!=null) ? fieldseparator :
            DefaultEmitConfiguration.INSTANCE.getGroupingPathSeparator();

        StringBuffer sb = AntXFixture.newStringBuffer();

        sb.append(indicatorLabel(e,iz,allowed));

        if ((allowed&MainField.INCLUDE_PROJECT)!=0) {
            sb.append(SEP);
            sb.append(projectLabelNoNull(e,iz,allowed));
        }
        if ((allowed&MainField.INCLUDE_TARGET)!=0) {
            sb.append(SEP);
            sb.append(targetLabelNoNull(e,iz,allowed));
        }
        if ((allowed&MainField.INCLUDE_TASK)!=0) {
            sb.append(SEP);
            sb.append(taskLabelNoNull(e,iz,allowed));
        }

        return sb.substring(0);
    }


    /**
     * Returns a best-guess indicator label. Never returns <i>null</i>.
     **/
    private String indicatorLabel(BuildEvent e, IndicatorZone iz, int mask)
    {
        String s = getLabel_(MainField.INDICATOR,e,iz);
        return (s!=null) ? s : iz.getValue();
    }


    /**
     * Returns a best-guess project category name. Returns <i>null</i> if
     * event not associated with any project(?!).
     **/
    private String projectLabel(BuildEvent e, IndicatorZone iz, int mask)
    {
        Project myproject = e.getProject();
        if (myproject==null) {
            return null;
        }
        String s = getLabel_(MainField.PROJECT,e,iz);
        if (s==null) {
            s = projectName(myproject);
        }
        return s;
    }


    /**
     * Returns a best-guess project category name. Never returns <i>null</i>.
     * @since JWare/AntX 0.4
     **/
    private String projectLabelNoNull(BuildEvent e, IndicatorZone iz, int mask)
    {
        String label = projectLabel(e,iz,mask);
        if (label==null) {
            label = uistrs().get("emit.unknown.project.grp");
        }
        return label;
    }


    /**
     * Returns a best-guess target category name. Returns <i>null</i> if
     * event not associated with any target.
     **/
    private String targetLabel(BuildEvent e, IndicatorZone iz, int mask)
    {
        Target target = e.getTarget();
        if (target==null) {
            return null;
        }
        String s = getLabel_(MainField.TARGET,e,iz);
        if (s==null) {
            s = target.getName();
            if (s==null) {
                s = "target@"+System.identityHashCode(target);
            }
        }
        return s;
    }


    /**
     * Returns a best-guess target category name. Never returns <i>null</i>.
     * @since JWare/AntX 0.4
     **/
    private String targetLabelNoNull(BuildEvent e, IndicatorZone iz, int mask)
    {
        String label = targetLabel(e,iz,mask);
        if (label==null) {
            label = uistrs().get("emit.unknown.target.grp");
        }
        return label;
    }


    /**
     * Returns a best-guess task category name. Returns <i>null</i> if
     * event not associated with any task.
     **/
    private String taskLabel(BuildEvent e, IndicatorZone iz, int mask)
    {
        Task task = e.getTask();
        if (task==null) {
            return null;
        }
        String s = getLabel_(MainField.TASK,e,iz);
        if (s==null) {
            if ((task instanceof Nameable) && (mask&MainField.INCLUDE_NESTED)!=0) {
                String name = ((Nameable)task).getName();
                if (name!=null) {
                    return name;
                }
            }
            if (task.getTaskName()==null) {
                return "task@"+System.identityHashCode(task);
            }
            return task.getTaskName();
        }
        return s;
    }


    /**
     * Returns a best-guess task category name. Never returns <i>null</i>.
     * @since JWare/AntX 0.4
     **/
    private String taskLabelNoNull(BuildEvent e, IndicatorZone iz, int mask)
    {
        String label = taskLabel(e,iz,mask);
        if (label==null) {
            label = uistrs().get("emit.unknown.task.grp");
        }
        return label;
    }

// ---------------------------------------------------------------------------------------

    private List m_items = AntXFixture.newList();
    private String m_Id;
    private Object m_referToMapper;
}

/* end-of-GroupingMapper.java */
