/**
 * $Id: CheckpointTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2003,2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.feedback;

import  java.io.PrintStream;

import  org.apache.tools.ant.types.Reference;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.starters.EchoThingTask;

/**
 * Task that emits a simple checkpoint to listening build monitors. Usually defined
 * as &lt;checkpoint&gt; The 'tofile' parameter is exclusive-- it <em>prevents</em>
 * the checkpoint from being broadcast to listening monitors.
 * <p>
 * <b>Examples:</b><pre>
 *    &lt;checkpoint/&gt;
 *    &lt;checkpoint msgid="cp.last.compiled"/&gt;
 *    &lt;checkpoint message="Checkstyle Warnings: ${N.warnings}"/&gt;
 *    &lt;checkpoint with="ec.metrics" tofile="${reports}/etc/profiled.last"/&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2003,2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,helper
 * @see      EmitTask
 **/

public class CheckpointTask extends EchoThingTask implements EmitConfigurable
{
    /**
     * Initializes a new CheckpointTask instance.
     **/
    public CheckpointTask()
    {
        super(AntX.feedback);
        setThingRefId("_");
    }


    /**
     * Returns the fallback emit configuration this task will
     * use to generate diagnostics emitters, etc. Never returns
     * <i>null</i>; usually an enclosing emit configure task.
     **/
    public EmitConfiguration getDefaults()
    {
        return EmitContext.getConfigurationNoNull();
    }


    /**
     * Assigns a primary emit configuration definition to this
     * task. This task will use this configuration instead of
     * any enclosing context configuration.
     * @param r the {@linkplain EmitConfiguration} reference
     **/
    public void setWith(Reference r)
    {
        require_(r!=null,"setWith- nonzro refid");
        m_withEC= r;
    }


    /**
     * Returns this task's user-defined emit configuration or
     * <i>null</i> if none defined.
     * @see #setWith
     **/
    public final Reference getWith()
    {
        return m_withEC;
    }



    /**
     * Tells this checkpoint whether the owning target's label
     * should be included in output message. Is on by default. Note
     * that target information is <em>only</em> addable to 
     * user-supplied messages; plain timestamps never include other
     * information.
     * @param include <i>false</i> to exclude source label
     * @since JWare/AntX 0.5
     **/
    public void setIncludeSource(boolean include)
    {
        m_includeSource = include;
    }



    /**
     * Returns <i>true</i> if this checkpoint will include its
     * owning target's name in emitted message. Defaults yes.
     * @since JWare/AntX 0.5
     **/
    public final boolean willIncludeSource()
    {
        return m_includeSource;
    }



    /**
     * Returns the effective EmitConfiguration used by this
     * task. Never returns <i>null</i>.
     * @see #setWith
     * @see #getDefaults
     * @.safety single
     **/
    protected synchronized EmitConfiguration getEmitHelper()
    {
        if (m_ECInstance!=null) {
            return m_ECInstance;
        }
        if (getWith()!=null) {
            m_ECInstance = (EmitConfiguration)getReferencedObject
                (getProject(), getWith().getRefId(), EmitConfiguration.class);
        }
        else {
            m_ECInstance = getDefaults();
        }
        return m_ECInstance;
    }


    /**
     * Emits the checkpoint line to either a file or any
     * listening log4j monitors.
     **/
    protected void echoThing()
    {
        final long NOW = System.currentTimeMillis();

        EmitConfiguration helper = getEmitHelper();

        String finalMsg, userMsg = getMsg();
        if (userMsg.length()!=0) {
            if (willIncludeSource()) {
                finalMsg = uistrs().
                    get("emit.checkpoint",
                        /*who*/getOwningTarget().getName(),
                        /*when*/helper.stampify(NOW),
                        /*what*/userMsg);
            } else {
                finalMsg = uistrs().
                    get("emit.checkpoint.nosource",
                        /*when*/helper.stampify(NOW),
                        /*what*/userMsg);
            }
        } else {
            finalMsg = helper.stampify(NOW);
        }

        if (getToFile()==null) {
            Emit.broadcast(helper.getEmitter(), finalMsg, null, NoiseLevel.INFO);
        } else {
            PrintStream out = new PrintStream(getOutputStream());
            out.println(finalMsg);
            out.flush();
            out.close();
        }
    }


    private Reference m_withEC;
    private EmitConfiguration m_ECInstance;
    private boolean m_includeSource=true;//NB:true=>backward-compatible pre AntX-0.5!
}

/* end-of-CheckpointTask.java */
