/**
 * $Id: BuildRule.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition.solo;

import  java.util.Iterator;
import  java.util.Stack;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.NoiseLevel;
import  com.idaremedia.antx.apis.BuildAssertionException;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.starters.StrictOuterTask;

/**
 * A shareable build rule type; usually defined as &lt;rule&gt;.
 * <p>
 * <b>Usage Notes</b>(also apply to {@linkplain RuleMacroDef rule macros})<b>:</b><ul>
 * <li>
 * Generally build rules should be declared at the project level to ensure
 * they're completely configured and added to the project's references before any task
 * needs/uses it.
 * <li>
 * The combine logic of a build rule is always a logical '<i>and</i>'; the rule's evaluation
 * is aborted on the first failed requirement. In other words, don't use rules to
 * create 'is-present' properties; use a real target, the &lt;condition&gt; task,
 * or the AntX &lt;tally&gt; task to implement this common function.
 * <li>
 * Empty rules always evaluate "<i>true</i>".
 * <li>
 * Build rules cannot refer-to other build rules (using the standard 'refid' attribute);
 * this would just be weird. Build rules are, by definition, the common bits; other
 * rules types (like asserts and prefers) refer to rules. Note a build rule's nested
 * conditions <em>can</em> refer to other build rules.
 * <li>
 * A build rule is composed of one kind-of condition: either requirements or preferences.
 * The two kinds cannot be mixed; this way the effect (on the build process) of a
 * failed evaluation can be predicted with reasonable certainty. &lt;prefer&gt; tasks
 * cannot refer to build rules whose effect is build failure. Likewise &lt;require&gt;
 * tasks cannot refer to build rules whose effect are warnings.
 * <li>
 * Be careful when specifying individual update properties for a rule's 'require'
 * items. If the enclosing rule is a global rule, these properties will be set <em>once</em>
 * in a particular project (since a Project's properties are write-once). This effect
 * shouldn't be problematic for requirements since the idea is to <b>fail</b> on a false
 * evaluation.
 * <li>
 * Be <em>really really</em> careful when specifying individual default properties for
 * a rule's 'prefer' items. If the preference evaluates false it will default the
 * property for the current project (always) <em>and for any child projects created
 * when using 'ant', 'antcall', 'call', 'callforeach' and 'foreach.'</em> You have been
 * warned. Having said that, this "velcro-ism" is often exactly the effect you want when
 * you use prefer with defaults-- some top-level target ensures the universe is happy
 * for all of the sub-build bits by enforcing their pre-requisites.
 * <li>
 * </ul>
 * <p>
 * <b>Rules vs. Rule-Macros?:</b><ul>
 * <li>
 * What's the difference between a rule and a rule macro? Well you would use a rule
 * macro if you wanted to parameterize a rule. A rule is a single entity; if you call
 * it from multiple sources (includes parallel threads) you're reusing a single
 * rule. Rule macros on the other hand, create a unique instance for each invocation.
 * <li>
 * Rules can define top-level parameters like failure messages, etc.; macros cannot.
 * <li>
 * Rule macros (<span class="src">&lt;ruledef&gt;</span>) are definition tasks and
 * can be included in standard antlib definition files (like
 * <span class="src">&lt;typedef&gt;</span> and
 * <span class="src">&lt;typedef&gt;</span>; rules cannot. You can assign unique
 * namespaces, etc. (Of course you an also import your common rules.)
 * </ul>
 * <p>
 * <b>Thread-safety Notes:</b><ul>
 * <li>
 * A build rule's thread-safety is partially dependent on a
 * subclass's agreement not to expose its conditions publicly (where they can be
 * modified unsafely). The first time a build rule is evaluated it freezes itself from
 * further modification (cannot add any more rules). Aside from this check at the
 * 'eval' method, concurrent evaluations incur no locking overhead in the nested
 * conditions.
 * <li>
 * A build rule's thread-safety is definitely dependent on the thread-safety of
 * the standard Ant conditions it supports. Even if the AntX build and other custom
 * conditions can be evaluated concurrently, some of the inherited Ant conditions
 * cannot. The &lt;available&gt; task/condition for example is not safe for concurrent
 * evaluation (for the correct eyeballs, see the toggling of an internal
 * '<code>isTask</code>' flag).
 * </ul>
 * <p>
 * <b>Examples:</b><pre>
 *  &lt;rule id="j2se14-available"&gt;
 *       &lt;require msgid="msg.no.j2se14"&gt;
 *          &lt;isclass name="java.lang.StrictMath"/&gt;
 *          &lt;isclass name="javax.naming.InitialContext"/&gt;
 *          &lt;isclass name="java.nio.Buffer"/&gt;
 *       &lt;/require&gt;
 *  &lt;/rule&gt;
 *
 *  &lt;rule id="apis-java-available" msgid="msg.missing.required.apis"&gt;
 *      &lt;require ruleid="j2se14-available"/&gt;
 *      &lt;require isclass="org.apache.log4j.Logger" msgid="msg.no.log4j.12"/&gt;
 *      &lt;require isclass="junit.framework.Assert" msgid="msg.no.junit.37"/&gt;
 *  &lt;/rule&gt;
 *
 *  &lt;rule id="doc-labels-available-internal"&gt;
 *      &lt;fixturecheck allset="module.name,module.version"/&gt;
 *      &lt;prefer isnotwhitespace="${module.doc.title}" msgid="warn.no.doc.title"
 *           property="module.doc.title"  default="${module.name} API Documentation"/&gt;
 *      &lt;prefer isnotwhitespace="${module.doc.header}" msgid="warn.no.doc.header"
 *           property="module.doc.header" default="${module.name}-${module.version}"/&gt;
 *      &lt;prefer isnotwhitespace="${module.doc.footer}" msgid="warn.no.doc.footer"
 *           property="module.doc.footer" defaultproperty="module.doc.header"/&gt;
 *  &lt;/rule&gt;
 * </pre>
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   api,infra
 * @see      AssertTask
 * @see      PreferTask
 **/

public class BuildRule extends RuleType implements StrictOuterTask
{
    /**
     * Initializes a new build rule.
     **/
    public BuildRule()
    {
        super(AntX.rules);
    }


    /**
     * Initializes a new CV-labeled build rule.
     * @param iam CV-label (non-null)
     **/
    public BuildRule(String iam)
    {
        super(iam);
    }


// ---------------------------------------------------------------------------------------
// Parameters, Nested Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Adds a requirement to this rule.
     * @return new requirement
     **/
    public BooleanRule createRequire()
    {
        Requirement requirement = new Requirement();
        includeRule(requirement);
        requirement.setEnclosingTask(this);
        return requirement;
    }


    /**
     * Adds a preference to this rule.
     * @return new preference
     **/
    public BooleanRule createPrefer()
    {
        Preference preference = new Preference();
        includeRule(preference);
        preference.setEnclosingTask(this);
        return preference;
    }


    /**
     * Adds a fixture check to this rule. This object is not
     * actually stored anywhere in this rule. Any false inlined condition
     * applied to the fixture check will trigger a build failure.
     * @return new fixturecheck
     **/
    public BooleanRule createFixtureCheck()
    {
        return new VerifyFixture(true/*force-fail-quick*/);
    }


    /**
     * Returns the supposed effect if this rule evaluates
     * <i>false</i>. Should be called after this rule is fully
     * configured.
     **/
    public NoiseLevel getFailureEffect()
    {
        return m_failureEffect;
    }


    /**
     * Adds a sub-rule to this build rule. If it is the first sub-rule
     * it determines the kind of subsequent sub-rules allowed. Otherwise,
     * the sub-rule must of the expected (dominant) type.
     * @param rule sub-rule to be added
     * @throws BuildException if rule of wrong kind
     **/
    private void includeRule(BooleanRule rule)
        throws BuildException
    {
        boolean isRequire = (rule instanceof Requirement);
        if (m_failureEffect!=null) {
            checkAcceptable(this,isRequire,m_failureEffect);
        } else if (isRequire) {
            m_failureEffect = NoiseLevel.ERROR;
            setStopQuickEvaluation(true);
        } else {
            m_failureEffect = NoiseLevel.WARNING;
            setStopQuickEvaluation(false);
        }
        xaddCondition(rule);
    }


    /**
     * Shared test for acceptability of nested requirements and
     * preferences. This logic must be shared by rule definitions and
     * rule macro definitions.
     * @param failureEffect the effect under test (non-null)
     **/
    static void checkAcceptable(Task from, boolean isRequire,
                                NoiseLevel failureEffect)
    {
        if ((isRequire  && (!failureEffect.isAsBadAs(NoiseLevel.ERROR))) ||
            (!isRequire && (failureEffect.isAsBadAs(NoiseLevel.ERROR)))) { //only nest one kind
            String id= from.getTaskName()!=null ? from.getTaskName() : "build-rules";
            String ermsg = AntX.uistrs().get("brul.only.onekind",id);
            from.log(ermsg, Project.MSG_ERR);
            throw new BuildException(ermsg, from.getLocation());
        }
    }

// ---------------------------------------------------------------------------------------
// Evaluation:
// ---------------------------------------------------------------------------------------

    /**
     * Ensures this rule instance does not contain any circular
     * references in its definition.
     * @param stk stack of referred-to (or used) rules (non-null)
     * @param clnt [optional] call controls
     * @throws BuildException if circular dependency discovered
     * @.impl sets latch after 1st successful call
     **/
    public void verifyNoCircularDependency(Stack stk, Requester clnt)
    {
        if (!m_circularityChecked) {
            if (!isEmpty()) {
                Iterator itr= getConditions().iterator();
                if (clnt==null) {
                    clnt = new Requester.ForComponent(this);
                }
                while (itr.hasNext()) {
                    Stack itemstk = new Stack();
                    itemstk.addAll(stk);
                    Object oc = itr.next();
                    if (oc instanceof BooleanRule) {
                        ((BooleanRule)oc).verifyNoCircularDependency(itemstk, clnt);
                    }
                    itemstk.clear();
                }
            }
            m_circularityChecked = true;
        }
    }



    /**
     * No-op; build rules must be explicitly evaluated via their
     * shareable condition APIs to ensure they're safe for calls
     * from multiple threads. Build rules aren't nestable.
     **/
    public final boolean eval()
    {
        //burp...loudly
        log(uistrs().get("brul.only.shareabl.apis"),Project.MSG_WARN);
        return false;
    }



    /**
     * Called to record this conditon's evaluation results. For requirements
     * a <i>false</i> result is very bad. <i>true</i> results are basically
     * ignored.
     * @.impl Copied from AssertTask, ssmc
     * @throws BuildException if effect is build failure and evaluated <i>false</i>
     **/
    protected void setEvalResult(boolean istrue, final String listing)
        throws BuildException
    {
        boolean likePrefer= likePrefer();

        if (!istrue) {
            String what = getUpdatePropertyValue();
            String prop;

            prop = getUpdateProperty();
            if (prop!=null) {
                if (likePrefer) {
                    getProject().setInheritedProperty(prop,what);
                } else {
                    checkIfProperty_(prop,true);
                    getProject().setNewProperty(prop,what);
                }

                log("Build rule (preference="+likePrefer+") was false; have set '"
                    +prop+"' property to "+what, Project.MSG_DEBUG);
            }

            prop = getUpdateVariable();
            if (prop!=null) {
                log("Build rule (preference="+likePrefer+") was false; setting '"
                    +prop+"' variable to "+what, Project.MSG_DEBUG);

                ExportedProperties.set(prop,what);
            }

            String errmsg;
            if (getMsgId()==null) {
                errmsg = uistrs().get("brul.failed",getId(),listing);
            } else {
                errmsg = getMsg(newMsgGetter(getId(),listing));
            }

            violationOccured(errmsg);

            if (!likePrefer) {
                throw new BuildAssertionException(errmsg, getLocation());
            }
        }
    }


    /**
     * Called whenever this rule's final evaluation is negative.
     * Subclasses can extend the default response which simply logs
     * an error message.
     * @param ermsg error message (non-null)
     **/
    protected void violationOccured(String ermsg)
    {
        if (ermsg==null) {
            ermsg= uistrs().get("brul.failed.default");
        }

        int noiselevel = getFailureEffect().isAsBadAs(NoiseLevel.ERROR)
            ? Project.MSG_ERR
            : BooleanRule.getPreferenceMsgLevel(getProject()).getNativeIndex();

        log(ermsg, noiselevel);
    }


    /**
     * Returns <i>true</i> if this rule is a preference-only based
     * rule. Looks at the failure effect of this rule; if it's anything
     * but a WARNING then it's not like a preference.
     * @since JWare/AntX 0.3
     **/
    protected final boolean likePrefer()
    {
        return !isStopQuickEvaluation();
    }


// ---------------------------------------------------------------------------------------
    private NoiseLevel m_failureEffect=null;//NB: unknown->defined by 1st subrule
}

/* end-of-BuildRule.java */
