/**
 * $Id: ShortHandValueURIHandler.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.InstanceFactory;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.starters.ValueURIHandlerSkeleton;

/**
 * Value URI handler that returns the string "<span class="src">true</span>" or
 * "<span class="src">false</span>" based on the inlined short hand condition's
 * result. Will return <i>null</i> if the named condition is not recognized. This
 * value URI handler is not installed by default; you must explicitly install
 * it as shown in the examples section.
 * <p/>
 * However you install this handler, you can only use the AntX-defined URI scheme
 * (condition) names. See the full declaration file
 * <span class="src">com/idaremedia/antx/valueuri/antlib.xml</span> for a listing 
 * of all the recognized short hand condition names (of course, you could just 
 * read this source file also).
 * <p/>
 * To allow you to specify the properties in the evaluated text, the short hand value
 * handler understands the alternate "<span class="src">@(propertyname)</span>"
 * format in the URI fragment; see example below.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;do true="${<b>$isantversion:</b>^.*version 1\.6.*"&gt;...
 *    &lt;domatch value="${<b>$allsettrue:</b>metrics.enabled,maven.enabled}"&gt;...
 *    &lt;property name="vars.enabled" value="${<b>$filenotempty:</b>@(varsfile)}"/&gt;
 *
 *   -- To Install --
 *    &lt;manageuris action="install"&gt;
 *       &lt;parameter name="isset"
 *             value="com.idaremedia.antx.condition.ShortHandValueURIHandler"/&gt;
 *       &lt;parameter name="isnotset"
 *             value="com.idaremedia.antx.condition.ShortHandValueURIHandler"/&gt;
 *       ...<i>[All other standard names and your own custom ones if desired]</i>
 *    &lt;/manageuris&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 * @see       URIable
 * @see       com.idaremedia.antx.condition.solo.ConditionValueURIHandler
 **/

public class ShortHandValueURIHandler extends ValueURIHandlerSkeleton
{
    /**
     * Set up the map of name to condition evaluators.
     */
    private static InstanceFactory m_IF= new InstanceFactory();
    static {
        m_IF.put("$isset:",IsSet.class);
        m_IF.put("$isnotset:",IsNotSet.class);
        m_IF.put("$notset:",IsNotSet.class);
        m_IF.put("$issettrue:",IsSetTrue.class);
        m_IF.put("$allset:",AllSet.class);
        m_IF.put("$allsettrue:",AllSetTrue.class);
        m_IF.put("$alltrue:",AllSetTrue.class);
        m_IF.put("$anyset:",AnySet.class);
        m_IF.put("$anysettrue:",AnySetTrue.class);
        m_IF.put("$anytrue:",AnySetTrue.class);
        m_IF.put("$noneset:",NoneSet.class);
        m_IF.put("$isbool:",IsBoolean.class);
        m_IF.put("$isboolean:",IsBoolean.class);
        m_IF.put("$isnumeric:",IsNumeric.class);
        m_IF.put("$isnumber:",IsNumeric.class);
        m_IF.put("$notwhitespace:",IsNotWhitespace.class);
        m_IF.put("$isnotwhitespace:",IsNotWhitespace.class);
        m_IF.put("$isref:",IsReference.class);
        m_IF.put("$isreference:",IsReference.class);
        m_IF.put("$isantversion:",IsAntVersion.class);
        m_IF.put("$filenotempty:",FileNotEmpty.class);
        m_IF.put("$isdirectory:",IsDirectory.class);
        m_IF.put("$isdir:",IsDirectory.class);
        m_IF.put("$isresource:",IsResource.class);
        m_IF.put("$isres:",IsResource.class);
    }



    /**
     * Initializes a new short hand value uri handler.
     * Only the standard condition names are supported by this
     * handler.
     * @see #addMapping addMapping(&#8230;)
     **/
    public ShortHandValueURIHandler()
    {
    }



    /**
     * Programmatic extension point that allows subclasses to add
     * own condition names and condition classes.
     * @param name protocol marker string (non-null)
     * @param claz URIable class (non-null)
     * @throws ClassCastException if <span class="src">claz</span> is not
     *  assign-compatible with {@linkplain URIable}.
     **/
    public static void addMapping(String name, Class claz)
    {
        AntX.require_(name!=null && claz!=null,
            AntX.conditions+"ValueURIHandler:", "addMapin- nonzro args");

        if (!URIable.class.isAssignableFrom(claz)) {
            throw new ClassCastException(claz.getName());
        }

        synchronized(m_IF) {
            m_IF.put(name,claz);
        }
    }



    /**
     * Common work that determines and evaluates the condition
     * represented by the value URI.
     * @param uriFragment resource information from uri (non-null)
     * @param fullUri the full value uri (non-null)
     * @param clnt caller information (non-null)
     * @return result or <i>null</i> if did not find match.
     **/
    protected Boolean eval(String uriFragment, String fullUri, Requester clnt)
    {
        int i= fullUri.indexOf(':');
        if (i>0) {
            String which= fullUri.substring(0,++i);
            if (i<fullUri.length()) {
                Object c= m_IF.newInstance(which);
                if (c instanceof URIable) {
                    clnt.getProject().setProjectReference(c);
                    if (c instanceof Task) {
                        Task t = (Task)c;
                        t.setLocation(clnt.getLocation());
                        t.init();
                    }
                    String fragment = fullUri.substring(i);
                    fragment = Tk.resolveString(clnt.getProject(),fragment,true);
                    ((URIable)c).xsetFromURI(fragment);
                    return ((URIable)c).eval() ? Boolean.TRUE : Boolean.FALSE;
                }
            }
        }
        return null;
    }



    /**
     * Returns either the string "<span class="src">true</span>" or
     * "<span class="src">false</span>" for a known condition. Otherwise
     * returns whatever the default value for an unrecognized URI would
     * be (usually <i>null</i>).
     * @return "true" or "false" if uri described a known condition,
     *         otherwise <i>null</i>.
     **/
    public String valueFrom(String uriFragment, String fullUri, Requester clnt)
    {
        Boolean B = eval(uriFragment, fullUri, clnt);
        if (B!=null) {
            return String.valueOf(B.booleanValue());
        }
        return getDefaultValue(fullUri,clnt);
    }
}


/* end-of-ShortHandValueURIHandler.java */