/**
 * $Id: FlexCondition.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.FlexValueSupport;

/**
 * A condition whose unknown value can be either an inlined literal value, a property's
 * value, a variable's value, or a reference's value. Property, variable, and reference
 * values are determined at evaluation time.  FlexConditions are useful when embedded
 * inside shareable build rules because their checked-values are determined when the
 * condition is evaluated not as the build file is parsed.
 *
 * @since    JWare/AntX 0.2
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   impl,infra
 **/

public abstract class FlexCondition extends AssertableProjectComponent
    implements Condition, FlexValueSupport
{
    /**
     * Initializes a new flex condition.
     **/
    protected FlexCondition()
    {
        super(AntX.conditions);
    }


    /**
     * Initializes a new CV-labeled flex condition.
     * @param iam CV-label (non-null)
     **/
    protected FlexCondition(String iam)
    {
        super(iam);
    }



    /**
     * Returns underlying flex value as-is. The returned value is
     * one of: a literal value (used as-is), a property name,
     * a variable name, or a reference identifier.
     **/
    public final String getFlexValue()
    {
        return getValueHelper().get();
    }



    /**
     * Returns flex value resolved of any property references. The
     * returned value is one of: a literal value (used as-is), a
     * property name, a variable name, or a reference identifier.
     * @since JWare/AntX 0.4
     **/
    public final String getResolvedFlexValue()
    {
        return getValueHelper().sourceString(getProject());
    }



    /**
     * Sets a property whose value is to be evaluated
     * by this condition. Property isn't read until this condition
     * is evaluated.
     * @param property the property's name (non-null)
     **/
    public void setProperty(String property)
    {
        setLiteral(property);
        getValueHelper().setIsProperty(true);
    }


    /**
     * Returns <i>true</i> if this condition is setup for a
     * property value check.
     **/
    public final boolean isProperty()
    {
        return getValueHelper().isProperty();
    }


    /**
     * Returns property name used by evaluation method. Returns
     * <i>null</i> if never set or value is an exported property.
     **/
    public final String getProperty()
    {
        return isProperty() ? getValueHelper().get() : null;
    }



    /**
     * Sets exported property name used by evaluation method.
     * Property's value determined at evaluation time.
     * @param exportedproperty the exported property's name (non-null)
     **/
    public void setVariable(String exportedproperty)
    {
        setLiteral(exportedproperty);
        getValueHelper().setIsExported(true);
    }


    /**
     * Synonym for {@linkplain #setVariable setVariable}.
     **/
    public final void setVar(String exportedproperty)
    {
        setVariable(exportedproperty);
    }


    /**
     * Returns <i>true</i> if this condition is setup for an
     * exported property check.
     **/
    public final boolean isVariable()
    {
        return getValueHelper().isExported();
    }


    /**
     * Returns the exported property name used by evaluation
     * method. Returns <i>null</i> if never set or value is a regular
     * property.
     * @see #isVariable
     **/
    public final String getVariable()
    {
        return isVariable() ? getValueHelper().get() : null;
    }



    /**
     * Sets a reference whose value is to be evaluated
     * by this condition. Reference isn't read until this condition
     * is evaluated.
     * @param refid the reference's name (non-null)
     **/
    public void setReference(String refid)
    {
        setLiteral(refid);
        getValueHelper().setIsReference(true);
    }


    /**
     * Returns <i>true</i> if this condition is setup for a
     * reference value check.
     **/
    public final boolean isReference()
    {
        return getValueHelper().isReference();
    }


    /**
     * Returns reference name used by evaluation method. Returns
     * <i>null</i> if never set or check isn't for a reference.
     **/
    public final String getReference()
    {
        return isReference() ? getValueHelper().get() : null;
    }



    /**
     * Sets this condition's to-be-evaluated literal string.
     * Subclass should determine under what parameter a literal
     * is exposed (for example, 'value' or 'name' or 'spam').
     * @param rawvalue value to be matched (non-null)
     **/
    public void setLiteral(String rawvalue)
    {
        getValueHelper().set(rawvalue);
        getValueHelper().setIsLiteral();//NB:clr all is-a-flags!
    }


    /**
     * Returns <i>true</i> if this condition is setup for an
     * literal (as-is) value check.
     **/
    public final boolean isLiteral()
    {
        return getValueHelper().isLiteral();
    }


    /**
     * Returns literal value used by this evaluation method or
     * <i>null</i> if never set or value is not a literal (to be used
     * as-is).
     **/
    public final String getLiteral()
    {
        return isLiteral() ? getValueHelper().get() : null;
    }



    /**
     * Verifies that this condition's value has been explicitly
     * defined.
     * @param required [optional] list of possible value parameters; use
     *                 "<i>null</i>" for default list
     * @throws BuildException if value not defined
     **/
    protected void verifyIsDefined_(String required)
    {
        if (getValueHelper().isUndefined()) {
            if (required==null) {
                required="property|variable|reference";
            }
            String ermsg = uistrs().get("task.needs.this.attr",
                                        getTypicalName(),required);
            log(ermsg,Project.MSG_ERR);
            throw new BuildException(ermsg);
        }
    }



    /**
     * Call to verify that this condition can proceed with evaluation.
     * By default ensures this condition is in a valid project and
     * has a defined value.
     * @param calr caller's identifier
     * @see #verifyIsDefined_
     * @throws BuildException if not in project or value not defined
     **/
    protected void verifyCanEvaluate_(String calr)
    {
        verifyInProject_(calr);
        verifyIsDefined_(null);
    }



    /**
     * Returns the underlying value's flex string for this
     * condition. Must never return <i>null</i>.
     **/
    protected abstract FlexString getValueHelper();


    /**
     * Returns this condition's typical name. Used in error messages.
     * By default returns lowercased version of this condition's class's
     * leaf name. For example, condition
     * "<code>com.idaremedia.antx.condition.IsNotSet</code>"is named
     * "<i>isnotset</i>".
     **/
    protected String getTypicalName()
    {
        return Tk.lowercaseFrom(Tk.leafNameFrom(getClass()));
    }
}

/* end-of-FlexCondition.java */
