/**
 * $Id: CheckSetCondition.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.condition;

import  java.util.Iterator;
import  java.util.List;
import  java.util.StringTokenizer;

import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.taskdefs.condition.Condition;

import  com.idaremedia.antx.AntXFixture;
import  com.idaremedia.antx.AssertableProjectComponent;
import  com.idaremedia.antx.ExportedProperties;
import  com.idaremedia.antx.FlexString;
import  com.idaremedia.antx.helpers.InnerString;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.ownhelpers.InnerProperties;
import  com.idaremedia.antx.parameters.Handling;
import  com.idaremedia.antx.parameters.IgnoreWhitespaceEnabled;
import  com.idaremedia.antx.parameters.IsA;
import  com.idaremedia.antx.parameters.MalformedCheckEnabled;
import  com.idaremedia.antx.parameters.PropertySource;
import  com.idaremedia.antx.parameters.TrimEnabled;

/**
 * Starting implementation for any condition that evaluates a set of nested
 * items (properties, variables, references, etc.) against some simple existence
 * condition.
 *
 * @since    JWare/AntX 0.2 (pulled up from AllSet AntX 0.5)
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  single
 * @.group   api,infra
 **/

public abstract class CheckSetCondition extends AssertableProjectComponent
    implements Condition, TrimEnabled, IgnoreWhitespaceEnabled, MalformedCheckEnabled
{
    /**
     * Initializes new set check condition instance.
     * @param iam CV-label (non-null)
     **/
    protected CheckSetCondition(String iam)
    {
        super(iam);
    }


    /**
     * Initializes a set check condition for a set of existing properties.
     * @param properties comma-delimited list of properties
     * @param P condition's project
     **/
    protected CheckSetCondition(String iam, String properties, final Project P)
    {
        super(iam);
        setProject(P);
        setProperties(properties);
    }



    /**
     * Marks this condition as a negation. Evaluates <i>true</i>
     * if <em>none</em> of the included items are set.
     **/
    protected void setNegate(boolean negate)
    {
        m_isNegate = negate;
    }


    /**
     * Returns <i>true</i> if this is a negative test.
     **/
    public final boolean isNegate()
    {
        return m_isNegate;
    }
    


    /**
     * Tells this conditon to look for items containing positive
     * boolean string values.
     * @since JWare/AntX 0.5
     **/
    protected final void setTruesOnly()
    {
        m_truesOnly = true;
    }
    
    
    
    /**
     * Returns whether this condition should check all candidate
     * items for a positive boolean string. Is false by default.
     * @since JWare/AntX 0.5
     **/
    protected final boolean isTruesOnly()
    {
        return m_truesOnly;
    }


// ---------------------------------------------------------------------------------------
// Parameters:
// ---------------------------------------------------------------------------------------

    /**
     * Sets property name updated by evaluation method.
     * @param property the property's name (non-null)
     **/
    public void setTrueProperty(String property)
    {
        m_updateProperty = property;
        m_isVar = false;
    }


    /**
     * Returns property name updated by evaluation method. Returns
     * <i>null</i> if never set or value is an exported property.
     **/
    public final String getTrueProperty()
    {
        return m_isVar ? null : m_updateProperty;
    }


    /**
     * Sets exported property name updated by evaluation method.
     * @param variable the exported property's name (non-null)
     **/
    public void setTrueVariable(String variable)
    {
        m_updateProperty = variable;
        m_isVar = true;
    }


    /**
     * Returns the exported property name update by evaluation method.
     * Returns <i>null</i> if never set or value is a regular property.
     * @see #isUpdateVariable
     **/
    public final String getTrueVariable()
    {
        return m_isVar ? m_updateProperty : null;
    }


    /**
     * Returns <i>true</i> if this condition is setup to update an
     * exported property on evaluation.
     **/
    public final boolean isUpdateVariable()
    {
        return m_isVar;
    }


    /**
     * Sets whether a property containing only whitespace is
     * considered "not-set."
     **/
    public void setWhitespace(Handling response)
    {
        m_ignoreWS= Handling.IGNORE.equals
            (Handling.simplifyIgnoreOrNot(response,Handling.ACCEPT));
    }


    /**
     * Returns <i>true</i> if property values of all whitespace
     * should be ignored (as if not set). Defaults <i>false</i>.
     **/
    public final boolean ignoreWhitespace()
    {
        return m_ignoreWS;
    }


    /**
     * Sets whether included item values will be trimmed before
     * considered "not-set."
     * @param trim <i>true</i> if trim item values
     **/
    public void setTrim(boolean trim)
    {
        m_willTrim = trim;
    }


    /**
     * Returns <i>true</i> if include values will be trimmed
     * before evaluated. Defaults to <i>false</i>.
     **/
    public final boolean willTrim()
    {
        return m_willTrim;
    }


    /**
     * Tells this condition whether properties must be completely
     * resolved to be considered "set". If the choice is
     * either "ignore" or "accept" then unresolved property
     * values are considered okidoki (set).
     * @since JWare/AntX 0.4
     **/
    public void setMalformed(Handling response)
    {
        // malformed="accept|ignore"
        if (Handling.isYes(response,Handling.REJECT)) {
            m_checkUnresolved= false;
        }
        // malformed="reject|balk|inherit"
        else {
            m_checkUnresolved= true;
        }
    }


    /**
     * Returns how this condition will handle unresolved
     * properties. Returnes either "accept" or "reject".
     * @since JWare/AntX 0.4
     **/
    public final Handling getMalformedHandling()
    {
        return m_checkUnresolved ? Handling.REJECT : Handling.ACCEPT;
    }


    /**
     * Short-hand for adding a list of properties to this condition.
     * @param properties comma-delimited list of properties
     **/
    public final void setProperties(String properties)
    {
        StringTokenizer st= new StringTokenizer(properties,",");
        InnerString holder= new InnerString();
        while (st.hasMoreTokens()) {
            holder.setValue(st.nextToken());
            includeItem(new FlexString(holder.toString(getProject()),true));
        }
    }


    /**
     * Short-hand for adding a nested pattern-based property
     * set to this condition. The property set's domain is limited
     * to the {@linkplain PropertySource#SCRIPT script}.
     * @param pattern the pattern against which all script
     *        properties checked (non-null)
     * @.sideeffect Turns malformed='reject' on automatically
     * @since JWare/AntX 0.4
     **/
    public final void setPropertiesLike(String pattern)
    {
        InnerProperties holder = new InnerProperties();
        holder.setProject(getProject());
        holder.setLike(pattern);
        holder.setDomain(PropertySource.SCRIPT.getValue());
        addConfiguredProperties(holder);
    }



    /**
     * Short-hand for adding a list of variables to this condition.
     * @param variables comma-delimited list of variables
     * @since JWare/AntX 0.5
     **/
    public final void setVariables(String variables)
    {
        StringTokenizer st= new StringTokenizer(variables,",");
        InnerString holder= new InnerString();
        while (st.hasMoreTokens()) {
            holder.setValue(st.nextToken());
            FlexString item = new FlexString(holder.toString(getProject()),false);
            item.setIsExported(true);
            includeItem(item);
        }
    }



    /**
     * Short-hand for adding a list of references to this condition.
     * @param references comma-delimited list of variables
     * @since JWare/AntX 0.5
     **/
    public final void setReferences(String references)
    {
        StringTokenizer st= new StringTokenizer(references,",");
        InnerString holder= new InnerString();
        while (st.hasMoreTokens()) {
            holder.setValue(st.nextToken());
            FlexString item = new FlexString(holder.toString(getProject()),false);
            item.setIsReference(true);
            includeItem(item);
        }
    }


// ---------------------------------------------------------------------------------------
// Nested Elements:
// ---------------------------------------------------------------------------------------

    /**
     * Returns name updated by evaluation method (either as property
     * or as variable). Returns <i>null</i> if neither an update property
     * nor an update variable have been set.
     **/
    protected final String getUpdateProperty()
    {
        return m_updateProperty;
    }


    /**
     * Returns this condition's list of nested items. Never returns
     * <i>null</i>.
     **/
    List getIncludes()
    {
        return m_includedItems;
    }


    /**
     * Returns a readonly iterator for this condition's list of
     * nested property definitions. Never returns <i>null</i>. The
     * returned iterator also returns contents of nested property
     * sets (in order) as flex values.
     * @since JWare/AntX 0.4
     **/
    protected final Iterator getIncludesIterator()
    {
        return new FlexIncludesIterator(new FlexOuterCondition() {
                public Project getProject() {
                    return CheckSetCondition.this.getProject();
                }
                public List getIncludes() {
                    return CheckSetCondition.this.getIncludes();
                }
                public boolean willTrim() {
                    return CheckSetCondition.this.willTrim();
                }
                public boolean ignoreWhitespace() {
                    return false;
                }
            });
    }


    /**
     * Adds the given item to this condition's set of items.
     * This object assumes ownership of flex string object.
     **/
    protected final void includeItem(FlexString s)
    {
        s.setProject(getProject());
        s.setTrim(willTrim());
        getIncludes().add(s);
    }


    /**
     * Adds the given properties declaration to this set.
     * Properties are determined at evaluation time. This object
     * assumes ownership of declaration.
     * @param decl properties declaration (non-null)
     * @since JWare/AntX 0.4
     **/
    public final void includeItem(InnerProperties decl)
    {
        getIncludes().add(decl);
    }


    /**
     * Adds a new property to this set. Property's value
     * will be determined at evaluation time.
     * @param holder property element
     **/
    public void addConfiguredProperty(InnerString holder)
    {
        require_(holder!=null,"addP- nonzro prop");
        includeItem(new FlexString(holder.toString(getProject()),true));
    }


    /**
     * Adds a new exported property to this set. Variable's
     * value will be determined at evaluation time.
     * @param holder property element
     **/
    public void addConfiguredVariable(InnerString holder)
    {
        require_(holder!=null,"addV- nonzro xprop");
        FlexString s = new FlexString(holder.toString(getProject()));
        s.setIsExported(true);
        includeItem(s);
    }


    /**
     * Synonym for {@linkplain #addConfiguredVariable
     * addConfiguredVariable}.
     **/
    public final void addConfiguredVar(InnerString holder)
    {
        addConfiguredVariable(holder);
    }


    /**
     * Adds a new reference to this set. Reference's value determined
     * at evaluation time.
     * @param holder reference element
     **/
    public void addConfiguredReference(InnerString holder)
    {
        require_(holder!=null,"addRef- nonzro refid");
        FlexString s = new FlexString(holder.toString(getProject()));
        s.setIsReference(true);
        includeItem(s);
    }


    /**
     * Adds a collection of properties to this set. Properties
     * are determined at evaluation time.
     * @param decl properties declaration (non-null)
     * @since JWare/AntX 0.4
     * @.sideeffect Turns malformed='reject' on automatically
     **/
    public void addConfiguredProperties(InnerProperties decl)
    {
        require_(decl!=null,"addProps- nonzro decl");
        includeItem(decl);
        setMalformed(Handling.REJECT);
    }


// ---------------------------------------------------------------------------------------
// Evaluation
// ---------------------------------------------------------------------------------------

    /**
     * Returns <i>true</i> if all of this condition's nested items have
     * been explicitly set in its project's environment. If this condition's
     * {@linkplain #setTruesOnly trues only} option has been set. Items must
     * exist and be equal to a positive boolean string.
     **/
    public boolean eval()
    {
        verifyInProject_("eval");

        final Project P = getProject();
        boolean notsets = isNegate();
        boolean istrue  = notsets ? true : false;
        int ALL = getIncludes().size();

        if (ALL!=0) {
            Boolean answer   = null;
            boolean ignoreWS = ignoreWhitespace();
            boolean iffTrue  = isTruesOnly();
            Iterator itr     = getIncludesIterator();
            int N=0;

            while (itr.hasNext()) {
                FlexString xv = (FlexString)itr.next();
                String v = null;

                if (xv.isProperty() && m_checkUnresolved) {
                    if (LocalPropertyExaminer.verifyProperty(P,xv)) {
                        v = xv.getValue(P);
                    } else {
                        log("CheckSet dropped '"+xv.get()+"' property because"+
                            " contains unresolved references", Project.MSG_DEBUG);
                    }
                } else {
                    v = xv.getValue(P);
                }
                N++;
                if (v!=null && ignoreWS && Tk.isWhitespace(v)) {
                    v = null;
                }
                if (v==null) {
                    if (notsets) { continue; }
                    answer= Boolean.FALSE;
                    break;
                } else if (notsets) {
                    if (!iffTrue || (Tk.string2PosBool(v)==Boolean.TRUE)) {
                        answer= Boolean.FALSE;
                        break;
                    }
                } else if (iffTrue) {
                    if (Tk.string2PosBool(v)!=Boolean.TRUE) {
                        answer= Boolean.FALSE;
                        break;
                    }
                }
            }
            if (answer!=null) {
                istrue = answer.booleanValue();
            } else {
                istrue = true/*N==ALL*/;
            }
        }

        if (istrue && m_updateProperty!=null) {
            String what = Strings.TRUE;
            if (isUpdateVariable()) {
                log("CheckSet was true; setting true-variable '"+m_updateProperty+"'",
                    Project.MSG_DEBUG);
                ExportedProperties.set(m_updateProperty,what);
            } else {
                log("CheckSet was true; setting true-property '"+m_updateProperty+"'",
                    Project.MSG_DEBUG);
                getProject().setNewProperty(m_updateProperty,what);
            }
        }

        return istrue;
    }



    /**
     * Common utility method for setting a checkset condition from a 
     * short hand value URI. To differentiate between a properties,
     * variables, etc&#46;, set the URI's query string to the correct
     * "isa" setting; for example:
     * <span class="src">$alltrue:@(list)?variable</span>.
     * @param uriFragment the value URI (resolved) (non-null)
     **/
    protected void xsetFromURIFragment(String uriFragment)
    {
        String list = uriFragment;
        IsA isa= IsA.PROPERTY;

        int i = uriFragment.lastIndexOf("?");
        if (i>0) {
            list = uriFragment.substring(0,i++);
            if (i<uriFragment.length()) {
                String isaName = uriFragment.substring(i);
                isa = IsA.from(isaName,isa);
            }
        }
        switch (isa.getIndex()) {
            case IsA.VARIABLE_INDEX: {
                setVariables(list);
                break;
            }
            case IsA.REFERENCE_INDEX:{
                setReferences(list);
                break;
            }
            default: {
                setProperties(list);
            }
        }
    }


    private String  m_updateProperty;
    private boolean m_isVar;
    private boolean m_isNegate;
    private boolean m_ignoreWS, m_willTrim;
    private List    m_includedItems= AntXFixture.newList(5);
    private boolean m_checkUnresolved;
    private boolean m_truesOnly;
}

/* end-of-CheckSetCondition.java */
