/**
 * $Id: InterpretLoggedTask.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  java.io.File;
import  java.io.FileReader;
import  java.io.IOException;
import  java.io.Reader;
import  java.io.StringReader;
import  java.util.Properties;

import  org.apache.tools.ant.AntTypeDefinition;
import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.ComponentHelper;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.AntX;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.ownhelpers.ConditionalInnerNameValuePair;
import  com.idaremedia.antx.ownhelpers.ConditionalParameters;
import  com.idaremedia.antx.parameters.IsA;
import  com.idaremedia.antx.parameters.TransformHelper;

/**
 * Helper task that evaluates a captured log for some pass/fail criteria and sets a set
 * of project properties in response.
 * <p>
 * <b>Example Usage:</b><pre>
 *     &lt;typedef name="javadoclog-checker" 
 *         classname="com.idaremedia.antx.capture.JavadocLogInterpreter"/&gt;
 *     &#8230;
 *     &lt;capturelogs importantfrom="warning"&gt;
 *         &lt;javadoc.../&gt;
 *         &lt;<b>evaluatelogged</b> prefix="_javadoc." 
 *              interpreter="javadoclog-checker"/&gt;
 *     &lt;/capturelogs&gt;
 *     &lt;domatch property="_javadoc.result"&gt;
 *         &lt;equals value="clean"&gt;
 *            ...
 *         &lt;otherwise&gt;
 *            ...
 *     &lt;/domatch&gt;
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class InterpretLoggedTask extends LogsUsingTask
{
    /**
     * Initializes a new log interpretation task.
     **/
    public InterpretLoggedTask()
    {
        super(AntX.capture+"InterpretLoggedTask:");
    }



    /**
     * Initializes this interpretation task's target log file.
     * Either a file URL or an actual file path can be defined. This log
     * file is used <em>instead of</em> any enclosing log recorder's
     * captured information.
     * @param fileref location of file (non-null)
     * @.impl Used for testing task.
     **/
    public void setLogFile(String fileref)
    {
        require_(fileref!=null,"setLogFile- nonzro fileref");
        m_fileRef = fileref;
        m_logFile = null;
    }



    /**
     * Sets the prefix of the property to set with the final conclusion. The
     * actually name of the property set will be
     * "<span class="src">&lt;prefix&gt;&#46;result</span>" and the value
     * will be one of: "clean", "problem", or "failure". If anything other than
     * a clean log is detected, this task will also set two count properties
     * "<span class="src">&lt;prefix&gt;&#46;errorcount</span>" and
     * "<span class="src">&lt;prefix&gt;&#46;warningcount</span>". If a prefix
     * is never set explicitly the value "<span class="src">_logcheck.</span>"
     * will be used.
     * @param propertyPrefix property name prefix (non-null)
     **/
    public void setPrefix(String propertyPrefix)
    {
        require_(propertyPrefix!=null && propertyPrefix.length()>0,
                 "setPrefix- nonzro property prefix");
        if (propertyPrefix.endsWith(".")) {
            m_propertiesPrefix = propertyPrefix;
        } else {
            m_propertiesPrefix = propertyPrefix+".";
        }
    }



    /**
     * Returns the name of the property this task will update with its
     * conclusion. Never returns <i>null</i> or the empty string.
     **/
    public final String getUpdateProperty()
    {
        return m_propertiesPrefix+"result";
    }



    /**
     * Returns the name of the property this task will update with the
     * number of errors. Never returns <i>null</i> or the empty
     * string.
     **/
    public final String getErrorCountProperty()
    {
        return m_propertiesPrefix+"errorcount";
    }



    /**
     * Returns the name of the property this task will update with the
     * number of warnings. Never returns <i>null</i> or the empty
     * string.
     **/
    public final String getWarningCountProperty()
    {
        return m_propertiesPrefix+"warningcount";
    }



    /**
     * Tells this task the type of the result set elements. Will default
     * to regular (write-once) project properties unless set to something
     * else.
     * @param resulttype type of results (variable|property|reference)
     **/
    public void setResultType(String resulttype)
    {
        require_(resulttype!=null,"setResultType- nonzro typename");
        IsA isa = IsA.from(resulttype);
        if (isa==null) {
            String err = getAntXMsg("task.illegal.param.value",getTaskName(),
                            resulttype, "resulttype");
            log(err, Project.MSG_ERR);
            throw new BuildException(err,getLocation());
        }
        m_resulttype = isa;
    }



    /**
     * Returns the preferred type of the result this task's interpreter
     * sets. Never returns <i>null</i>. Defaults IsA.PROPERTY.
     **/
    public final IsA getResultType()
    {
        return m_resulttype;
    }



    /**
     * Tells this task the maximum number of errors to tolerate before
     * a failure is recorded. Note that <em>any</em> errors will force
     * this task to conclude there was at least a <i>problem</i>; the
     * max threshold indicates when a problem turns into a failure.
     * @param maxErrors the maximum number of errors to tolerate (anything
     *        over is a failure).
     **/
    public void setMaxErrors(int maxErrors)
    {
        require_(maxErrors>=0,"setMaxErrors- zero or more value");
        m_maxErrors = maxErrors;
    }



    /**
     * Returns the maximum number of errors this task will tolerate before
     * recording a failure. This value is zero (0) unless defined otherwise.
     **/
    public final int getMaxErrors()
    {
        return m_maxErrors;
    }



    /**
     * Tells this task the maximum number of warnings to tolerate before
     * a failure is recorded. Note that <em>any</em> warnings will force
     * this task to conclude there was at least a <i>problem</i>; the
     * max threshold indicates when a problem turns into a failure.
     * @param maxWarnings the maximum number of errors to tolerate (anything
     *        over is a failure).
     **/
    public void setMaxWarnings(int maxWarnings)
    {
        require_(maxWarnings>=0,"setMaxWarnings- zero or more value");
        m_maxWarnings = maxWarnings;
    }



    /**
     * Returns the maximum number of warnings this task will tolerate before
     * recording a failure. This value is ten (10) unless defined otherwise.
     **/
    public final int getMaxWarnings()
    {
        return m_maxWarnings;
    }



    /**
     * Tells this task the name of the actual log interpreter algorithm to
     * use. The incoming string is either a typedef'ed application interpreter
     * or the fully qualified class name of the interpreter.
     * @param typename name of the interpreter as declared with &lt;typedef&gt;
     **/
    public void setInterpreter(String typename)
    {
        require_(typename!=null,"setInterprter- nonzro typename");

        ComponentHelper ch = ComponentHelper.getComponentHelper(getProject());
        AntTypeDefinition atd = ch.getDefinition(typename);
        if (atd!=null) {
            Object impl = atd.create(getProject());
            if (impl instanceof LogInterpreter) {
                m_logInterpreter = (LogInterpreter)impl;
            }
        }
        if (m_logInterpreter==null) {
            try {
                Class c = Class.forName(typename);
                if (LogInterpreter.class.isAssignableFrom(c)) {
                    m_logInterpreter = (LogInterpreter)c.newInstance();
                }
            } catch(Exception anyX) {
                log(anyX.getMessage(),Project.MSG_ERR);
            }
        }
        if (m_logInterpreter==null) {
            String err = getAntXMsg("task.bad.custimpl.class1",typename,
                LogInterpreter.class.getName());
            log(err, Project.MSG_ERR);
            throw new BuildException(err, getLocation());
        }
    }



    /**
     * Adds a single interpreter parameter. The new parameter
     * is merged with any existing parameters.
     * @param arg the parameter information (non-null)
     **/
    public void addConfiguredParameter(ConditionalInnerNameValuePair arg)
    {
        if (m_interpreterArgs==null) {
            m_interpreterArgs = new ConditionalParameters(getProject());
        }
        m_interpreterArgs.addConfiguredParameter(arg);
    }



    /**
     * Ensure both a non-existing result property and a log interpreter have
     * been specified for this task.
     * @throws BuildException if incomplete task configuration.
     **/
    protected void verifyCanExecute_(String calr)
    {
        super.verifyCanExecute_(calr);

        if (m_logInterpreter==null) {
            String err = getAntXMsg("task.needs.this.attr",getTaskName(),
                "interpreter");
            log(err, Project.MSG_ERR);
            throw new BuildException(err, getLocation());
        }

        switch(getResultType().getIndex()) {
            case IsA.PROPERTY_INDEX:  {
                checkIfProperty_(getUpdateProperty(),false);
                break;
            }
            case IsA.REFERENCE_INDEX: {
                checkIfReference_(getUpdateProperty(),true);
                break;
            }
        }

        if (m_fileRef!=null) {
            m_logFile = new File(TransformHelper.toPath(m_fileRef,getProject()));
        }
    }



    /**
     * Returns instructions that should be passed to interpreter. Never
     * returns <i>null</i>.
     **/
    private Properties getInstructionsNoNull()
    {
        Properties ini;
        if (m_interpreterArgs==null) {
            ini = new Properties();
        } else {
            ini = m_interpreterArgs.copyOfSimpleKeyValues(getProject(),true);
        }
        return ini;
    }



    /**
     * Evaluates either a named log file or the information captured by
     * the nearest logs recorder and updates one or more result 
     * project properties.
     * @throws BuildException if task improperly defined, unable to read log file,
     *          or interpreter signals error.
     */
    public void execute()
    {
        verifyCanExecute_("exec");

        Reader inputr = null;
        
        if (m_logFile!=null) {
            if (!m_logFile.exists()) {
                getProject().setNewProperty(getUpdateProperty(),LogInterpreter.CLEAN);
                return;
            }
            try {
                inputr = new FileReader(m_logFile);
            } catch(IOException ioX) {
                throw new BuildException(ioX);
            }
        } else {
            inputr = new StringReader(getVUTLog());
        }

        m_logInterpreter.interpret(inputr, new ConfigWrap(this,getInstructionsNoNull()));
    }



    /**
     * Returns <i>true</i> always. We can accept log files as well as
     * log recorders for our input data.
     */
    protected boolean isSourceFlexible()
    {
        return true;
    }


    private String m_fileRef;
    private File m_logFile;
    private String m_propertiesPrefix = "_logcheck.";
    private int m_maxErrors = 0;
    private int m_maxWarnings = 10;
    private LogInterpreter m_logInterpreter;
    private ConditionalParameters m_interpreterArgs;
    private IsA m_resulttype= IsA.PROPERTY;
}




/**
 * Adapter of InterpretParameters to InterpretLoggedTask's interface.
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    impl,helper
 **/
class ConfigWrap extends Requester.ForComponent implements InterpretParameters {
    private final InterpretLoggedTask m_task;
    private final Properties m_argv;
    ConfigWrap(InterpretLoggedTask task, Properties argv) {
        super(task);
        m_task = task;
        m_argv = argv;
    }
    public String getErrorCountProperty() {
        return m_task.getErrorCountProperty();
    }
    public int getMaxErrors() {
        return m_task.getMaxErrors();
    }
    public int getMaxWarnings() {
        return m_task.getMaxWarnings();
    }
    public String getUpdateProperty() {
        return m_task.getUpdateProperty();
    }
    public String getWarningCountProperty() {
        return m_task.getWarningCountProperty();
    }
    public boolean updateProperties() {
        return true;
    }
    public IsA getResultType() {
        return m_task.getResultType();
    }
    public Properties getInstructions() {
        return m_argv;
    }
}

/* end-of-InterpretLoggedTask.java */