/**
 * $Id: EmmaTextResultsInterpreter.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx.capture;

import  java.io.BufferedReader;
import  java.io.IOException;
import  java.io.Reader;

import  org.apache.tools.ant.BuildException;

import  com.idaremedia.antx.helpers.Tk;

/**
 * Our default Ant &lt;emma/report&gt; text output log interpreter. This interpreter
 * assumes that the text output is for all levels (up to method) sorted in the default
 * order using weighted units. Results are determined by first looking at what EMMA
 * says passed. Each "no pass" metric is interpreted as a single error. If a coverage
 * metric passes EMMA's criteria, it is checked for our warning criteria which is 75%
 * or better coverage. Any coverage below 75% is considered a warning.
 * <p/>
 * <b>Example Usage:</b><pre>
 *    &lt;typedef name="emma-checker"
 *       classname="com.idaremedia.antx.capture.EmmaTextResultsInterpreter"/&gt;
 *    ...
 *    &lt;emma enabled="yes"&gt;
 *       &lt;report sourcepath="${java.sources}"&gt;
 *          &lt;fileset refid="emma.files"/&gt;
 *          &lt;txt outfile="coverage.txt" depth="all"/&gt;
 *       &lt;/report&gt;
 *    &lt;/emma&gt;
 *    ...
 *    &lt;evaluatelogged logfile="coverage.txt" <b>interpreter="emma-checker"</b>
 *        maxerrors="0" maxwarnings="1" prefix="emma."/&gt;
 *    &lt;domatch property="emma.result"&gt;
 *       &lt;equals value="CLEAN"&gt;
 *          ...
 *       &lt;otherwise&gt;
 *    ...
 *
 * </pre>
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   single
 * @.group    api,helper
 **/

public final class EmmaTextResultsInterpreter implements LogInterpreter
{
    private static final String HEADER0= "[class, %]\t[method, %]\t[block, %]\t[line, %]";

    /**
     * Initializes a new EMMA text result file interpreter.
     **/
    public EmmaTextResultsInterpreter()
    {
        super();
    }


    private int beginning(String s, int end)
    {
        int start = end;
        int i = end;
        while (i>=0) {
            if (!Character.isDigit(s.charAt(i))) {
                break;
            }
            start = i;
            i--;
        }
        return start;
    }


    private boolean below75percent(String s, int resultend)
    {
        int i = s.lastIndexOf('%',resultend);
        if (i>0) {
            int j = i;
            i = beginning(s,j-1);
            int percent = Tk.integerFrom(s.substring(i,j),Tk.NO_INT);
            if (percent!=Tk.NO_INT) {
                return percent<75;
            }
        }
        return false;
    }


    public String interpret(Reader inputr, InterpretParameters config)
        throws BuildException
    {
        String result = PROBLEM;
        int nUnderCovered = 0;
        int nBelow75 = 0;

        try {
            BufferedReader in = new BufferedReader(inputr,512);
            String line;
            while ((line=in.readLine())!=null) {
                if (line.startsWith(HEADER0)) {
                    String resultsline = in.readLine();
                    if (resultsline!=null) {
                        result = CLEAN;
                        int i,from=resultsline.length()-1;
                        while(from>0) {
                            i= resultsline.lastIndexOf(')',from);
                            if (i<0) {
                                break;
                            }
                            if (resultsline.charAt(i+1)=='!') {
                                nUnderCovered++;
                            } else if (below75percent(resultsline,i)) {
                                nBelow75++;
                            }
                            from = i-1;
                            i = resultsline.lastIndexOf('(',from);
                            if (i>0) {
                                from = i-1;
                            }
                        }
                    }
                    break;
                }
            }
            in.close();
        } catch(IOException ioX) {
            throw new BuildException(ioX);
        }

        if (nUnderCovered!=0) {
            if (nUnderCovered>config.getMaxErrors()) {
                result = FAILURE;
            } else {
                result = PROBLEM;
            }
        }
        else if (nBelow75!=0) {
            if (nBelow75>config.getMaxWarnings()) {
                result = FAILURE;
            } else {
                result = PROBLEM;
            }
        }

        ResultsHelper.set(config,nUnderCovered,nBelow75,result);
        return result;
    }
}

/* end-of-EmmaTextResultsInterpreter.java */