/**
 * $Id: Problems.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Task;

import  com.idaremedia.antx.apis.ProblemHandler;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.apis.ScriptLocatable;

/**
 * Administers problem handlers and other error-related fixture information. Usually
 * a default fixture problem handler is installed by an early initialization build
 * target. Local (scoped) problem handlers are not addressed because the application
 * can install scoped handlers via the AntX <span class="src">&lt;protect&gt;</span>
 * flowcontrol taskset.
 *
 * @since    JWare/AntX 0.3
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,infra
 * @see      ProblemHandler
 * @see      com.idaremedia.antx.apis.Responses
 **/

public class Problems implements FixtureCore, FixtureAdministrator
{
    /**
     * The fixture component id for all problem administration.
     **/
    public static final String FXID= FixtureIds.PROBLEM_ADMINISTRATION;



    /**
     * Problem handler that silently ignores the problem. Not null.
     **/
    public static final ProblemHandler IgnoringHandler=
        new ProblemHandler() {
                public void problem(Object nugget, int nl) {
                    //nada
                }
            };




    /**
     * Returns the location of a helper component or its client if
     * helper has no location. Never returns <i>null</i>.
     * @param calr requester (non-null)
     * @param impl helper's location if known
     * @since JWare/AntX 0.5
     **/
    public static final Location getHelperLocation(Task impl, Requester calr)
    {
        Location l = impl.getLocation();
        if (l==Location.UNKNOWN_LOCATION) {
            l = calr.getLocation();
        }
        return l==null ? Location.UNKNOWN_LOCATION : l;
    }



    /**
     * Returns the location of a calling client or the service if
     * client is anonymous. Never returns <i>null</i>.
     * @param impl the helper (non-null)
     * @param calr the client (non-null)
     * @since JWare/AntX 0.5
     **/
    public static final Location getCallerLocation(Requester calr, ScriptLocatable impl)
    {
        Location l = calr.getLocation();
        if (l==Location.UNKNOWN_LOCATION) {
            l = impl.getLocation();
        }
        return l==null ? Location.UNKNOWN_LOCATION : l;
    }



    /**
     * Returns the current thread's ProblemHandler if one exists. Returns
     * <i>null</i> if no ProblemHandler explicitly installed for thread.
     **/
    public static ProblemHandler getDefaultHandler()
    {
        return (ProblemHandler)FixtureOverlays.getContextInstance().nearest(FXID);
    }



    /**
     * Helper to {@linkplain #getDefaultHandler getDefaultHandler} that
     * returns the {@linkplain #IgnoringHandler} if the current thread
     * has no explicit handler. Never returns <i>null</i>.
     **/
    public static ProblemHandler getDefaultHandlerNoNull()
    {
        ProblemHandler ph = getDefaultHandler();
        return (ph!=null) ? ph : IgnoringHandler;
    }



    /**
     * Installs a new default ProblemHandler for the current thread. This handler
     * becomes the active (and only administered) problem handler until it is
     * unwound or another handler is installed.
     * @param ph the new (default) problem handler (non-null)
     * @param noInstallHandler [optional] used to notify if cannot install
     * @return previous problem handler if any (can be <i>null</i>)
     * @throws BuildException if incoming handler already on iteration stack
     **/
    public static ProblemHandler installDefaultHandler(ProblemHandler ph,
                                                  ProblemHandler noInstallHandler)
    {
        if (ph==null) {
            throw new IllegalArgumentException
                (AntX.uistrs().get("problems.null.handlr"));
        }
        return (ProblemHandler)FixtureOverlays.installIfNot
                            (FXID, ph, noInstallHandler, "ProblemsHandler");
    }



    /**
     * Removes the most recently installed ProblemHandler for the current
     * thread. The previous installation is reactivated, or if this was
     * the only problem handler, the current thread's uncaught problem
     * handler is not undefined.
     * @param noUninstallHandler [optional]used to notify if cannot uninstall
     * @see com.idaremedia.antx.apis.Responses
     **/
    public static void unwindDefaultHandler(ProblemHandler noUninstallHandler)
    {
        FixtureOverlays.uninstallIfIs(FXID,noUninstallHandler,"ProblemsHandler");
    }



    /**
     * Installs test-harness cleanup method to clear up the various problem
     * handling fixture components. Your application should never use this
     * helper directly; it is provided so that test harnesses can reset the
     * environment to a known state.
     * @since JWare/AntX 0.4
     **/
    static {
        AntXFixture.setKillMethod
            (FXID,
             new String[]{"problems","problemhandlers"},
             new KillMethodSkeleton(FXID, "ProblemsHandler"));
    }


    /** Disallow; only public static utility methods. **/
    private Problems()
    { }
}

/* end-of-Problems.java */
