/**
 * $Id: FixtureOverlays.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2002-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.ProblemHandler;

import  java.util.HashMap;
import  java.util.Iterator;
import  java.util.Map;
import  java.util.Stack;

/**
 * Administrator of thread-local stacks of iteration settings. As the run context
 * moves through different targets and tasks certain settings need to move between
 * these entities independent of the current Ant project instance. Since Ant tasks
 * like &lt;antcall&gt; create new child project instances it is difficult for
 * a configuration task to declare a setting at build-iteration level (attaching
 * properties to the project may or may not propagate the settings properly depending
 * on how the &lt;antcall&gt; is declared). FixtureOverlays transcends projects, targets,
 * and tasks and is the foundation for many AntX configuration tasks.
 * <p/>
 * FixtureOverlays tries to be very tolerant to support blind-resets when tasks are
 * unwinding from an exception condition. FixtureOverlays should <em>not</em> be used
 * to store permanent, iteration independent, information because overlays will be
 * torn down for each new iteration regardless of whether the enclosing Ant runtime
 * remains intact (for example if builds are being launched from an always on 
 * "Ant console").
 * <p/>
 * Classes that use fixture overlays usually provide a public administration facade
 * utility (like {@linkplain UISMContext}) and an associated kill method for execution
 * harnesses.
 *
 * @since    JWare/AntX 0.1 (originally Iteration)
 * @author   ssmc, &copy;2002-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,infra
 * @see      KillMethod
 **/

public final class FixtureOverlays implements FixtureCore, FixtureAdministrator, Cloneable
{
    /**
     * Initializes a new empty overlays object.
     **/
    FixtureOverlays()
    { }



    /**
     * Returns a <em>deep</em> clone of this overlays instance. The returned
     * copy is completely independent of this instance, so changing its
     * callstack has no effect on this one.
     **/
    public synchronized Object clone()
    {
        try {
            FixtureOverlays copy = (FixtureOverlays)super.clone();
            copy.m_Env = (HashMap)m_Env.clone();
            Iterator itr = m_Env.entrySet().iterator();
            while (itr.hasNext()) {
                Map.Entry mE= (Map.Entry)itr.next();
                copy.m_Env.put(mE.getKey(),((Stack)mE.getValue()).clone());
            }
            copy.m_invalidCalls=0;
            return copy;
        } catch(CloneNotSupportedException clnx) {
            throw new InternalError(AntX.uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }



    /**
     * Returns the current ("frontmost") setting associated with given 
     * category. Returns <i>null</i> if category has no current setting.
     * @param category settings category (like 'UISM' or 'Logs', etc.)
     **/
    public synchronized Object nearest(Object category)
    {
        Stack all = callstack(category);
        if (all==null || all.isEmpty()) {
            return null;
        }
        return all.peek();
    }



    /**
     * Installs a new ("frontmost") setting for the given overlay
     * category. The new setting cannot be <i>null</i>. The method
     * returns the previous setting, providing a single locked get/set
     * operation.
     * @param category variable category (like 'UISM' or 'Emitter', etc.)
     * @param current the most-recent setting in category (non-null)
     * @return previous category setting (can be null)
     * @see FixtureIds
     **/
    public synchronized Object install(Object category, Object current)
    {
        Object previous = null;
        Stack all = callstack(category);

        if (all==null) {
            all = create(category);
        } else if (!all.isEmpty()) {
            previous = all.peek();
        }

        all.push(current);
        return previous;
    }



    /**
     * Zaps the most recent setting for a category from this overlays.
     * No-op if category has no active settings.
     * @param category settings category (like 'UISM' or 'Logs', etc.)
     * @return <i>true</i> if unwound occured, else <i>false</i>
     **/
    public synchronized boolean unwind(Object category)
    {
        boolean popped = true;
        Stack all = callstack(category);

        if (all!=null && !all.isEmpty()) {
            all.pop();
        } else {
            m_invalidCalls++;
            popped = false;
        }
        return popped;
    }



    /**
     * Clears all settings for a particular category from this overlays.
     * Your application should never use this method directly; it
     * is provided for overlays clients to provide a kill method to the
     * execution or test harness.
     * @param category settings category (non-null)
     * @return <i>true</i> if category cleared
     * @since JWare/AntX 0.4
     **/
    public synchronized boolean clear(Object category)
    {
        Stack all = callstack(category);
        if (all!=null) {
            all.clear();
            return true;
        }
        return false;
    }



    /**
     * Returns <i>true</i> if an object is already installed on a
     * category's stack. Useful for categorys that require a single instance
     * of any particular object on the stack at any one time. The check
     * verifies object equality by using the incoming object's 'equals' method.
     **/
    public synchronized boolean isInstalled(Object category, Object current)
    {
        if (current==null) {
            throw new IllegalArgumentException("isInstalled- NULL object");
        }
        Stack all = callstack(category);
        if (all!=null) {
            return all.contains(current);
        }
        return false;
    }



    /**
     * Installs an object into a category iff it doesn't already exist
     * on category's current thread stack. Common template method for many
     * iteration-based fixture administrators. Usually called by category
     * adminstrator's public-facing install method.
     * @param category variable category
     * @param next next visible category setting (non-null)
     * @param noInstallHandler [optional] callback if unable to install
     * @param what who's calling (used in any error messages)
     * @since JWare/AntX 0.3
     * @throws BuildException if unable to install category object
     **/
    public static Object installIfNot(Object category, Object next,
                                      ProblemHandler noInstallHandler,
                                      String what)
    {
        FixtureOverlays overlays = FixtureOverlays.getContextInstance();
        synchronized(overlays) {
            if (overlays.isInstalled(category,next)) {
                String error = AntX.uistrs().get("context.install.once",what);
                if (noInstallHandler!=null) {
                    noInstallHandler.problem(error,Project.MSG_ERR);
                }
                throw new BuildException(error);
            }
            return overlays.install(category,next);
        }
    }



    /**
     * Removes the most recently installed object for a particular category.
     * The previous installation is reactivated, or if this was only object,
     * the current thread's category object becomes undefined. Common template
     * method for many iteration-based administrators. Usually called by
     * category administrator's public-facing uninstall method.
     * @param category variable category
     * @param noUninstallHandler [optional] callback if unable to uninstall
     * @param what who's calling (used in any error messages)
     * @since JWare/AntX 0.3
     **/
    public static void uninstallIfIs(Object category,
                                     ProblemHandler noUninstallHandler,
                                     String what)
    {
        FixtureOverlays overlays = FixtureOverlays.getContextInstance();
        if (!overlays.unwind(category) && noUninstallHandler!=null) {
            String warning = AntX.uistrs().get("context.all.uninstalled",what);
            noUninstallHandler.problem(warning,Project.MSG_WARN);
        }
    }



    /**
     * Hook for error-reporting. Necessary to determine if inappropriate
     * unwinds have been done.
     **/
    public synchronized int numberInvalidUnwinds()
    {
        return m_invalidCalls;
    }



    /**
     * Returns the stack used to a particular category. Returns <i>null</i>
     * if not such stack.
     **/
    private Stack callstack(Object category)
    {
        if (category==null) {
            throw new IllegalArgumentException("callstack- NULL category");
        }
        return (Stack)m_Env.get(category);
    }



    /**
     * Factory method to create a category's settings stack. Automatically
     * installs the stack into this iteration context's map.
     **/
    private Stack create(Object category)
    {
        Stack all = new Stack();
        m_Env.put(category, all);
        return all;
    }


    private HashMap m_Env = new HashMap(31,0.85f);
    private int m_invalidCalls;


    //------------------------------------------------------------------------------------------|

    /**
     * Thread-based handle to a single FixtureOverlays instance. Child threads are given their
     * own independent copies of the handled iteration instance. Changes to the child's
     * context will not affect its parent.
     *
     * @since    JWare/AntX 0.1
     * @author   ssmc, &copy;2002-2003 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
     * @version  0.5
     * @.safety  guarded
     *
     **/
    public static final class Handle extends InheritableThreadLocal
    {
        /**
         * Creates new thread-local handle to an iteration.
         **/
        public Handle()
        {
            super();
        }

        /**
         * A spanking new FixtureOverlays instance.
         **/
        protected Object initialValue()
        {
            return new FixtureOverlays();
        }

        /**
         * A spanking new FixtureOverlays clone of parent's value.
         **/
        protected Object childValue(Object parentValue)
        {
            return ((FixtureOverlays)parentValue).clone();
        }
    }


    //------------------------------------------------------------------------------------------|


    /**
     * Returns the current thread's iteration settings. This thread-specific
     * variable is shared by all objects within the same thread. Usually, a class's
     * implementation provides an administrator to manage its context-specific
     * information.
     * @see FixtureIds
     * @see FixtureAdministrator
     **/
    public static final FixtureOverlays getContextInstance()
    {
        return (FixtureOverlays)Iteration.fixtureOverlays().get();
    }
}

/* end-of-FixtureOverlays.java */
