/**
 * $Id: FixtureExaminer.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2004-2005,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.lang.reflect.Method;
import  java.util.Iterator;
import  java.util.Map;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.PropertyHelper;
import  org.apache.tools.ant.UnknownElement;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.GenericParameters;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.PropertySource;

/**
 * Collection of utilities for extracting bits of information from the current
 * project and/or build-iteration fixture.
 *
 * @since    JWare/AntX 0.4 (Pulled from various sources)
 * @author   ssmc, &copy;2004-2005,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,helper
 * @see      com.idaremedia.antx.ownhelpers.LocalTk LocalTk
 * @.stereo  utility
 **/

public final class FixtureExaminer implements FixtureCore
{
    private static final String IAM_= AntX.AntX+"FixtureExaminer:";


    private static Location prethrow(Project project, Requester from,
                                     String message, int level)
    {
        Location loc = Location.UNKNOWN_LOCATION;
        if (from!=null) {
            from.log(message, level);
            loc = from.getLocation();
        } else {
            project.log(message,level);
        }
        return loc;
    }


    /**
     * Returns a type-checked referenced object. Common utility that
     * generates a resource bundle based messages if reference broken.
     * @param project the source project
     * @param from [optional] the calling element
     * @param refid the referred-to thing's identifier (non-null)
     * @param requiredClass the required class of referred-to thing (non-null)
     * @throws BuildException if no such reference or object is not compatible
     **/
    public static final Object getReferencedObject(Project project,
                                                   Requester from,
                                                   String refid,
                                                   Class requiredClass)
    {
        AntX.require_(refid!=null,IAM_,"getRefObj- nonzro id");
        AntX.require_(project!=null,IAM_,"getRefObj- nonzro project");

        Object o = project.getReference(refid);
        String error = null;

        if (o==null) {
            error = AntX.uistrs().get("task.missing.refid", refid);
        }
        else if (!requiredClass.isInstance(o)) {
            error = AntX.uistrs().get("task.bad.refid", refid,
                                      requiredClass.getName(),
                                      o.getClass().getName());
        }
        if (error!=null) {
            Location loc = prethrow(project,from,error,Project.MSG_ERR);
            throw new BuildException(error,loc);
        }
        return o;
    }



    /**
     * Returns a referenced object. Common utility that generates a
     * resource bundle based message if reference undefined.
     * @param project the source project
     * @param from [optional] the calling element
     * @param refid the referred-to thing's identifier (non-null)
     * @throws BuildException if no such reference
     **/
    public static final Object getReferencedObject(Project project,
                                                   Requester from,
                                                   String refid)
    {
        AntX.require_(refid!=null,IAM_,"getRefObj- nonzro id");
        AntX.require_(project!=null,IAM_,"getRefObj- nonzro project");

        Object value = project.getReference(refid);
        if (value==null) {
            String error = AntX.uistrs().get("task.missing.refid", refid);
            Location loc = prethrow(project,from,error,Project.MSG_ERR);
            throw new BuildException(error,loc);
        }

        return value;
    }



    private static String isOneOf(String value, String t0, String t1)
    {
        if (value.startsWith(t0)) {
            return value.substring(t0.length());
        }
        if (value.startsWith(t1)) {
            return value.substring(t1.length());
        }
        return null;
    }



    /**
     * Utility that converts a reference to a properties collection to a
     * standard Java <span class="src">Properties</span> object. Currently
     * handles references to <span class="src">GenericParameters</span>,
     * <span class="src">PropertySet</span>s, and
     * <span class="src>Properties</span> objects
     * @param project source project (non-null)
     * @param refid referenced data object (non-null)
     * @param defaultProperties [optional] set to a fallback properties if
     *         reference does not exist.
     * @return Properties object if reference is convertible; <i>null</i>
     *         if it is not; <span class="src">defaultProperties</span> if
     *         no such reference.
     * @since JWare/AntX 0.5
     **/
    public static Properties getReferencedProperties(Project project,
        String refid, Properties defaultProperties)
    {
        AntX.require_(refid!=null,IAM_,"getRefPropertiesObj- nonzro refid");
        AntX.require_(project!=null,IAM_,"getRefPropertiesObj- nonzro project");

        Properties Ps = null;
        Object o = project.getReference(refid);
        if (o!=null) {
            //NB: Extract properties against given project to ensure we
            //    resolve all ${} and @() placeholders!
            if (o instanceof GenericParameters) {
                Ps = ((GenericParameters)o).copyOfSimpleKeyValues(project);
            } else if (o instanceof PropertySet) {
                Ps = ((PropertySet)o).getProperties();
            } else if (o instanceof Properties) {
                Ps = ((Properties)o);
            }
        } else {
            Ps = defaultProperties;
        }
        return Ps;
    }



    /**
     * Looks for a top-level property or a nested property (inside a
     * global reference properties-like holder.
     * @param key the property lookuup key (non-null)
     * @param project the source project (non-null)
     * @return property value or <i>null</i> if undefined
     * @since JWare/AntX 0.5
     **/
    private static String getProperty(String key, Project project)
    {
        String value = project.getProperty(key);
        int i;
        if (value==null && (i=key.indexOf("->"))>0) {
            String holder = key.substring(0,i);
            i += 2;
            if (i<key.length()) {
                key = key.substring(i);
                Properties Ps = getReferencedProperties(project,holder,null);
                if (Ps!=null) {
                    value = Ps.getProperty(key);
                }
            }//!malformed
        }
        return value;
    }



    /**
     * Determine if given value uri prefix is one of the
     * AntX-reserved ones. Note that case is ignored when checking
     * the prefix.
     * @param name the script-supplied prefix (non-null)
     * @return <i>true</i> if reserved
     * @since JWare/AntX 0.5
     **/
    public static boolean isBuiltinValueURIScheme(String name)
    {
        if (name.charAt(0)=='$') {
            int n = name.length();
            int i = name.indexOf(':');
            if (i>0) {
                n = i;
            }
            name = name.substring(1,n);
        }

        boolean reserved = false;
        name = Tk.lowercaseFrom(name);

        switch(name.charAt(0)) {
            case 'r': {
                if ("ref".equals(name) || "reference".equals(name)) {
                    reserved = true;
                }
                break;
            }
            case 'v': {
                if ("var".equals(name) || "variable".equals(name)) {
                    reserved = true;
                }
                break;
            }
            case 'p': {
                if ("prop".equals(name) || "property".equals(name)) {
                    reserved = true;
                }
                break;
            }
            case 'i': {
                if ("itprop".equals(name)|| "itproperty".equals(name)) {
                    reserved = true;
                }
                break;
            }
        }
        return reserved || "null".equals(name);
    }



    /**
     * Looks for an iteration property or a nested iteration property.
     * @param key the property lookuup key (non-null)
     * @param project [optional] the source project
     * @return property value or <i>null</i> if undefined
     * @since JWare/AntX 0.5
     **/
    public static String getIterationProperty(String key, Project project)
    {
        Object value = Iteration.getProperty(key);
        int i;
        if (value==null && (i=key.indexOf("->"))>0) {
            String holder = key.substring(0,i);
            i += 2;
            if (i<key.length()) {
                key = key.substring(i);
                value = Iteration.getProperty(holder,key);
            }
        }
        return Iteration.strictStringifier().stringFrom(value,project);
    }



    /**
     * Determine if there is an installed value URI interpreter
     * in the given project.
     * @param project the source project (non-null)
     * @return <i>true</i> if installed, else <i>false</i>
     * @since JWare/AntX 0.5
     * @see com.idaremedia.antx.solo.ValueURIManagerTask
     **/
    public static boolean valueURIInterpreterInstalled(Project project)
    {
        AntX.require_(project!=null,IAM_,"valRdrInstal- nonzro project");
        Object interpreter = project.getReference(FixtureIds.VALUEURI_INTERPRETER);
        if (interpreter instanceof PropertyHelper) {
            return true;
        }
        return false;
    }



    /**
     * Determine the value of a standalone (unpiped) value uri. This
     * method is used by the public API which parses for piped variants.
     * @param value full value uri (single,non-null)
     * @param obof originating property being determined (non-null)
     * @param clnt call controls (non-null)
     * @return property or value uri results or null if cannot determine.
     **/
    private static final String funcValue(String value, String obof,
                                          Requester clnt)
    {
        String dflt=null,key;
        Project project = clnt.getProject();

        //Hard-coded; cannot override...
        if ((key = isOneOf(value,"$var:","$variable:"))!=null) {
            int i = key.lastIndexOf('?');
            if (i>0) {
                dflt = key.substring(i+1);
                key = key.substring(0,i);
            }
            return ExportedProperties.readstring(key,dflt);
        }
        if ((key = isOneOf(value,"$ref:","$reference:"))!=null) {
            Object o = project.getReference(key);
            return Iteration.lenientStringifer().stringFrom(o,project);
        }
        if ((key = isOneOf(value,"$itprop:","$itproperty:"))!=null) {
            return getIterationProperty(key,project);
        }
        if ((key = isOneOf(value,"$prop:","$property:"))!=null) {
            int i = key.lastIndexOf('?');
            if (i>0) {
                dflt = key.substring(i+1);
                key = key.substring(0,i);
            }
            if (!key.equals(obof)) {//NB:zap circular references!
                String match = getProperty(key,project);
                return match==null ? dflt : match;
            }
        }
        //User-coded; whatever...
        int i = value.indexOf(":");
        if (i>1) {
            String scheme = value.substring(1,i);
            return Iteration.valueURIHandler(scheme).valueFrom
                (value.substring(i+1), value, clnt);
        }
        return null;
    }



    /**
     * Converts any nested value URIs in given value. Called by AntX's
     * value uri interpreter and explicitly by a select set of
     * components like &lt;overlay&gt;.
     * @param value the encoded value to lookup (prefixed w/ '$type:')
     * @param obof [optional] originating property source (used to
     *             ensure no circular definitions in property lookups)
     * @param clnt originating call controls (non-null)
     * @since JWare/AntX 0.5
     * @see ValueURIHandler
     **/
    public static final String findValue(String value, String obof,
                                         Requester clnt)
    {
        AntX.require_(value!=null,IAM_,"getFlexValue- nonzro values");
        AntX.require_(clnt!=null && clnt.getProject()!=null,IAM_,
                "getFlexValue- nonzro project");

        //@since AntX 0.5 (supports custom valueuri handlers and pipes)
        String result = null;
        while (value.length()>2 && value.charAt(0)=='$') {
            int colon = value.indexOf(":");
            if (colon <= 1) {
                break;
            }
            String scheme = value.substring(1,colon);
            String singleuri = value;
            int piped = value.indexOf("|$",colon+1);
            String fragment = null;
            if (piped>colon) {
                fragment = value.substring(colon+1,piped);
                singleuri = "$"+scheme+":"+fragment;
            } else {
                fragment = value.substring(colon+1);
            }
            //clnt.log("Evaluating value-uri'"+singleuri+"'");
            String nextresult = funcValue(singleuri,obof,clnt);
            result = nextresult;
            if (result==null || singleuri==value) {
                break;
            }
            colon = value.indexOf(':',piped+2);
            if (colon<0) {
                result = null;//NB:malformed
                break;
            }
            String nextvalue = "$"+value.substring(piped+2,colon+1)+result;
            if (colon<value.length()-1) {
                nextvalue += value.substring(colon+1);
            }
            value = nextvalue;
            //clnt.log("  Next value-uri is '"+value+"'");
        }
        return result;
    }



    /**
     * Like {@linkplain #findValue(String,String,Requester)
     * findValue(String,String,&#8230;)} for anonymous utility-based
     * lookups.
     * @param project the source project (non-null)
     * @param value the encoded value to lookup (prefixed w/ '$type:')
     * @param obof [optional] originating property source (used to
     *             ensure no circular definitions in property lookups)
     * @since JWare/AntX 0.4
     **/
    public static final String findValue(Project project,
                                         String value, String obof)
    {
        return findValue(value,obof,new Requester.ForProject(project));
    }



    /**
     * Check whether a project property already exists and optionally
     * barfs if it does. Always emits a warning (at least) if the property
     * does exist. Most useful for verifying that a user-specified update
     * property doesn't already exist.
     * @param project the source project
     * @param from [optional] the calling element
     * @param property name of property to check
     * @param warn <i>true</i> if only issue a warning <em>otherwise
     *             can fail</em>
     * @return <i>true</i> if property already exists
     * @throws BuildException if property exists and isn't a warning
     * @since JWare/AntX 0.3
     **/
    public static boolean checkIfProperty(Project project, Requester from,
                                          String property, boolean warn)
    {
        AntX.require_(project!=null,IAM_,"chkProp- nonzro project");
        AntX.require_(property!=null,IAM_,"chkProp- nonzro nam");

        String it = Tk.getTheProperty(project,property);
        if (it!=null) {
            String msg = AntX.uistrs().get("task.warn.property.exists",property);
            Location loc = prethrow(project, from, msg,
                                    (warn ? Project.MSG_WARN : Project.MSG_ERR));
            if (!warn) {
                throw new BuildException(msg,loc);
            }
        }
        return it!=null;
    }



    /**
     * Check whether a project reference already exists and optionally
     * barfs if it does. Always emits a warning (at least) if the
     * reference does exist. Most useful for verifying that a
     * user-specified update reference doesn't already exist.
     * @param project the source project
     * @param from [optional] the calling element
     * @param refid name of reference to check
     * @param warn <i>true</i> if only issue a warning <em>otherwise
     *             can fail</em>
     * @return <i>true</i> if reference already exists
     * @throws BuildException if reference exists and isn't a warning
     **/
    public static boolean checkIfReference(Project project, Requester from,
                                           String refid, boolean warn)
    {
        AntX.require_(project!=null,IAM_,"chkRef- nonzro project");
        AntX.require_(refid!=null,IAM_,"chkRef- nonzro refid");

        Object it = project.getReference(refid);
        if (it!=null) {
            String msg = AntX.uistrs().get("task.warn.refid.exists",refid);
            Location loc = prethrow(project, from, msg,
                                    (warn ? Project.MSG_WARN : Project.MSG_ERR));
            if (!warn) {
                throw new BuildException(msg,loc);
            }
        }
        return it!=null;
    }



    /**
     * Special value object used to indicate a reference that
     * has yet to be determined (in other target, or out-of-scope).
     * @.pattern Fowler.SpecialValue
     * @since JWare/AntX 0.4
     **/
    public static final Object IGNORED_REFERENCE= "__ooscope_ref__";



    /**
     * Special value object used to indicate a reference that
     * could not be determined (generated errors).
     * @.pattern Fowler.SpecialValue
     * @since JWare/AntX 0.4
     **/
    public static final Object PROBLEM_REFERENCE= "__problem_ref__";



    /**
     * Special value object used to indicate a reference that
     * is about to be assigned by a subsequent factory task. Allows
     * the dynamic creation of reference id to appear atomic to
     * using tasks.
     * @.pattern Fowler.SpecialValue
     * @since JWare/AntX 0.5
     **/
    public static final Object PENDING_REFERENCE= "__pending_ref__";


    /**
     * Returns a reference only if that reference has already been
     * determined by the project. This method tries to differentiate
     * references that are defined inside out-of-scope targets or
     * after this caller's location within the current target.
     * @param P the project used to lookup references (non-null)
     * @param id the reference id (non-null)
     * @throws IllegalArgumentException if either parameter is <i>null</i>
     * @return {@linkplain #IGNORED_REFERENCE} if reference out of scope<br/>
     *    {@linkplain #PROBLEM_REFERENCE} if unable to retrieve reference<br/>
     *    <i>null</i> if no such reference in project
     * @since JWare/AntX 0.4
     **/
    public static Object trueReference(Project P, String id)
    {
        AntX.require_(P!=null && id!=null, IAM_, "trueRef- nonzro args");

        Object ref=null;
        Object ht = P.getReferences();
        Boolean B=null;
        Method m=null;
        try {
            //Ant 1.6|1.7 internal way
            if (AntX.ANT_PRE17) {
                m = ht.getClass().getMethod("getReal",GET_REAL_METHOD);
            } else {
                m = ht.getClass().getDeclaredMethod("getReal",GET_REAL_METHOD);
            }
            B = m.isAccessible() ? Boolean.TRUE : Boolean.FALSE;
            m.setAccessible(true);
            ref = m.invoke(ht, new Object[]{id});
            if (ref instanceof UnknownElement) {
                ref = IGNORED_REFERENCE;
            }
        } catch(Exception reflectX) {
            P.log("Unable to call 'Project$AntRefTable.getReal': "+reflectX.getMessage());
            //Fallback: Public way
            try {
                ref = P.getReference(id);
            } catch(Exception lookupX) {
                P.log(lookupX.getMessage(), Project.MSG_WARN);
                ref = PROBLEM_REFERENCE;
            }
        } finally {
            if (m!=null && B!=null & !B.booleanValue()) {
                try {
                    m.setAccessible(false);
                } catch(Exception anyX) {
                    /*burp*/
                }
            }
        }
        return ref;
    }



    /**
     * Returns whether the referenced item exists, has been configured,
     * and can be read. Complement to {@linkplain #trueReference trueReference}.
     * @param P the project used to lookup references (non-null)
     * @param id the reference id (non-null)
     * @since JWare/AntX 0.5
     **/
    public static boolean usableReference(Project P, String id)
    {
        Object ref = trueReference(P,id);
        if (ref==IGNORED_REFERENCE || ref==PROBLEM_REFERENCE
            || ref==PENDING_REFERENCE) {
            ref= null;
        }
        return ref!=null;
    }



    /**
     * Utility to convert a property source to a set of properties.
     * @param domain source of properties (non-null)
     * @param P [optional] project from which properties read (if needed)
     * @throws IllegalArgumentException if domain is <i>null</i>
     * @throws NullPointerException if project needed and not defined
     * @since JWare/AntX 0.4
     **/
    public static Map copyOfProperties(PropertySource domain, Project P)
    {
        AntX.require_(domain!=null,IAM_,"getProps- nonzro domain");

        Map mp=null;//shutup-compiler

        switch (domain.getIndex()) {
            case PropertySource.ALL_INDEX:
            case PropertySource.ALLlite_INDEX: {
                mp = P.getProperties();
                if (domain.getIndex()!=PropertySource.ALL_INDEX) {
                    removeImmutableJavaProperties(mp);
                }
                break;
            }
            case PropertySource.COMMAND_INDEX: {
                mp = P.getUserProperties();
                break;
            }
            case PropertySource.CONTROL_INDEX: {
                Project dumi= new Project();
                P.copyInheritedProperties(dumi);
                mp = dumi.getProperties();
                dumi = null;//gc
                break;
            }
            case PropertySource.SCRIPTlite_INDEX:
            case PropertySource.SCRIPT_INDEX: {
                //NB:inherited properties are also user properties!
                mp = P.getProperties();
                mp.keySet().removeAll(P.getUserProperties().keySet());
                if (domain.getIndex()!=PropertySource.SCRIPT_INDEX) {
                    removeImmutableJavaProperties(mp);
                }
                break;
            }
            case PropertySource.FIXTURE_INDEX: {
                mp = AntXFixture.copyOfProperties();
                break;
            }
            case PropertySource.JRE_INDEX: {
                mp = (Map)System.getProperties().clone();
                break;
            }
            default: {
                AntX.verify_(false,IAM_,"init- known domain");
            }
        }

        return mp;
    }



    /**
     * Removes the "classic" immutable java runtime properties
     * including anything that begins with "java&#46;", and
     * "sun&#46;".
     * @throws UnsupportedOperationException if
     *         <span class="src">map</span> is a readonly map.
     * @since JWare/AntX 0.4
     **/
    public static void removeImmutableJavaProperties(Map map)
    {
        AntX.require_(map!=null,IAM_,"zapSystemProps- nonzro map");

        Iterator itr= map.keySet().iterator();
        while (itr.hasNext()) {
            String s = itr.next().toString();
            if (s.startsWith("java.") || s.startsWith("sun.")) {
                itr.remove();
                continue;
            }
            for (int i=0;i<KEEPERS.length;i++) {
                if (KEEPERS[i].equals(s)) {
                    itr.remove();
                    continue;
                }
            }
        }
    }



    /**
     * Always look for these; do not allow them to be
     * removed in custom list.
     **/
    private static final String[] KEEPERS= {
        "os.name",   "os.arch",   "os.version",
        "user.name", "user.home", "user.dir",
        "user.language", "user.timezone", "user.country", "user.variant",
        "file.separator", "path.separator",
        "line.separator"
    };




    private static final Class[] GET_REAL_METHOD= {Object.class};


    /** Only permit generic utility methods. **/
    private FixtureExaminer()
    {
    }
}

/* end-of-FixtureExaminer.java */
