/**
 * $Id: ErrorSnapshot.java 186 2007-03-16 13:42:35Z ssmc $
 * Copyright 2002-2004 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your option) any later
 * version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                           EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.util.Hashtable;
import  java.util.Iterator;
import  java.util.List;
import  java.util.Map;
import  java.util.Properties;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Location;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;
import  org.apache.tools.ant.types.PropertySet;

import  com.idaremedia.antx.apis.FlexStringFriendly;
import  com.idaremedia.antx.apis.Nameable;
import  com.idaremedia.antx.helpers.Strings;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Struct class that keeps the environment and information about a thrown build exception.
 * Works with the AntX build-feedback modules and protected tasksets.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5
 * @.safety  guarded
 * @.group   impl,helper
 * @.stereo  struct
 **/

public final class ErrorSnapshot extends AssertableProjectComponent
    implements Cloneable, Nameable, FlexStringFriendly
{
    /**
     * Do common task-based initialization.
     **/
    private void init(Task source)
    {
        require_(source!=null,"ctor- nonzro task");
        setProject(source.getProject());
        m_taskName   = source.getTaskName();
        m_targetName = source.getOwningTarget().getName();
        setLocation(source.getLocation());
        m_Id = "ErrorSnapshot@"+System.identityHashCode(this);
    }


    /**
     * Creates a new standalone error snapshot.
     * @since JWare/AntX 0.4
     **/
    public ErrorSnapshot()
    {
        super(AntX.flow+"protect");
    }


    /**
     * Creates a new error snapshot.
     * @param source task creating snapshot (non-null)
     **/
    public ErrorSnapshot(Task source)
    {
        super(AntX.flow+"protect");
        init(source);
    }


    /**
     * Creates a new error snapshot of thrown exception.
     **/
    public ErrorSnapshot(Task source, Exception thrown)
    {
        super(AntX.flow+"protect");
        init(source);
        setThrown(thrown);
    }


    /**
     * Returns this snapshot's identifying name. Will return
     * this object's system identifier if never set. Never
     * returns <i>null</i>.
     **/
    public String getName()
    {
        return m_Id;
    }


    /**
     * Sets this snapshot's name explicitly. Usually done by
     * at construction. Cannot be set <i>null</i>.
     **/
    public void setName(String name)
    {
        if (name!=null) {
            m_Id = name;
        }
    }


    /**
     * Returns name of the task that created this snapshot.
     * Returns the empty string if this snapshot created standalone.
     **/
    public String getTaskName()
    {
        return m_taskName;
    }


    /**
     * Changes the name of task associated with this snapshot.
     * Useful if snapshot rendered by one task on behalf of another.
     **/
    public void setEffectiveTaskName(String taskName)
    {
        require_(taskName!=null,"setTskNam- nonzro nam");
        m_taskName = taskName;
    }


    /**
     * Returns name of the owning target of the task that
     * created this snapshot. Returns empty string if this snapshot
     * created standalone.
     **/
    public String getTargetName()
    {
        return m_targetName;
    }


    /**
     * Changes the name of target associated with this snapshot.
     * Useful if snapshot rendered by one target on behalf of another.
     **/
    public void setEffectiveTargetName(String targetName)
    {
        require_(targetName!=null,"setTrgtNam- nonzro nam");
        m_targetName = targetName;
    }


    /**
     * Sets this snapshot's <i>causing</i> exception.
     * @param thrown the thrown exception (non-null)
     **/
    public synchronized void setThrown(Exception thrown)
    {
        m_thrown = thrown;
        if (getLocation()==null && (thrown instanceof BuildException)) {
            setLocation(((BuildException)thrown).getLocation());
        }
    }


    /**
     * Sets this snapshot's <i>causing</i> exception and updates
     * its location to match the exception's.
     * @param thrown the thrown exception (non-null)
     * @param updateLoc <i>true</i> if snapshot's location should
     *        be sync'ed
     **/
    public synchronized void setThrown(Exception thrown,
                                       boolean updateLoc)
    {
        m_thrown = thrown;
        if (updateLoc && (thrown instanceof BuildException)) {
            setLocation(((BuildException)thrown).getLocation());
        }
    }


    /**
     * Returns this snapshot's exception. Can return <i>null</i> if
     * never defined.
     **/
    public Exception getThrown()
    {
        return m_thrown;
    }


    /**
     * Sets this snapshot task's location.
     **/
    public void setLocation(Location location)
    {
        m_location = location;
    }


    /**
     * Returns this snapshot task's location. Never returns
     * <i>null</i> but can return unknown location.
     **/
    public Location getLocation()
    {
        return m_location;
    }


    /**
     * Associates a local source comment with this snapshot.
     **/
    public void setComment(String comment)
    {
        m_comment = (comment==null) ? "" : comment;
    }


    /**
     * Returns this snapshot's comment string. Will return
     * empty string if never set.
     **/
    public final String getComment()
    {
        return m_comment;
    }


    /**
     * Adds a set of properties en-masse to this snapshot.
     * @param ps the property set to be added (as-is)
     * @since JWare/AntX 0.4
     **/
    public final void addConfiguredPropertySet(PropertySet ps)
    {
        require_(ps!=null,"addPropSet- nonzro propset");
        try {
            Properties p= ps.getProperties();
            synchronized(this) {
                if (m_env==null) {
                    m_env = p;
                } else {
                    getEnv().putAll(p);
                    p.clear();
                }
            }
            p=null;
        } catch(RuntimeException rtX) {
            log(uistrs().get("task.bad.propset",rtX.getMessage()),
                Project.MSG_ERR);
        }
    }


    /**
     * Returns this snapshot's captured properties. Never
     * returns <i>null</i>.
     **/
    protected synchronized Hashtable getEnv()
    {
        if (m_env==null) {
            m_env = new Hashtable(31,0.8f);
        }
        return m_env;
    }


    /**
     * Adds a captured property to this snapshot.
     * @param name property's name (non-null)
     * @param value property's value (non-null)
     **/
    public void setProperty(String name, String value)
    {
        if (value==null) {
            value=Strings.NULL;
        }
        getEnv().put(name,value);
    }


    /**
     * Saves a project property to this snapshot.
     * @param name the project property to be captured (non-null)
     **/
    public void captureProperty(String name)
    {
        String value = getProject().getProperty(name);
        if (value==null) {
            value= Strings.NULL;
        }
        getEnv().put(name,value);
    }


    /**
     * Saves a set of project properties to this snapshot.
     * @param namesList comma-delimited list of property names (non-null)
     **/
    public void captureProperties(String namesList)
    {
        List keys = Tk.splitList(namesList,",");
        for (int i=0,N=keys.size();i<N;i++) {
            captureProperty((String)keys.get(i));
        }
    }



    /**
     * Saves a script-supplied set of properties to this
     * snapshot. This method is a adaption point for saving
     * custom property sets to a snapshot. Call must protect
     * incoming properties against concurrent modification.
     * @since JWare/AntX 0.4
     **/
    public void captureProperties(Map properties)
    {
        getEnv().putAll(properties);
    }


    /**
     * Assumes a set of project properties as this snapshot's
     * base set.
     * @param fullP the full set of properties (user|all)
     **/
    private void assumeProperties(Hashtable fullP)
    {
        if (m_env==null) {
            m_env= fullP;
        } else {
            Hashtable byhand= m_env;
            m_env= fullP;
            m_env.putAll(byhand);//mine wins if is existin key
        }
    }


    /**
     * Saves all user properties to this snapshot.
     **/
    public synchronized void captureUserProperties()// *cough*
    {
        assumeProperties(getProject().getUserProperties());
    }


    /**
     * Saves all project properties to this snapshot.
     **/
    public synchronized void captureAllProperties()// *cough*
    {
        assumeProperties(getProject().getProperties());
    }


    /**
     * Returns new copy of this snapshot's current set of
     * properties.
     **/
    public synchronized Properties copyOfProperties()
    {
        Hashtable myP = getEnv();
        synchronized(myP) {
            if (myP instanceof Properties) {
                return (Properties)myP.clone();
            }
            Properties copiedP= new Properties();
            if (!myP.isEmpty()) {
                Iterator itr= myP.entrySet().iterator();
                while (itr.hasNext()) {
                    Map.Entry mE= (Map.Entry)itr.next();
                    copiedP.setProperty(mE.getKey().toString(),
                                        mE.getValue().toString());
                }
            }
            return copiedP;
        }
    }


    /**
     * Creates a clone of this snapshot (used if as a reference I
     * am inherited and passed to another independent project).
     **/
    public synchronized Object clone()
    {
        try {
            ErrorSnapshot copy= (ErrorSnapshot)super.clone();
            copy.m_env = copyOfProperties();//ok even if Properties
            return copy;
        } catch(CloneNotSupportedException clnx) {
            throw new InternalError(uistrs().get(AntX.CLONE_BROKEN_MSGID));
        }
    }


    /**
     * Creates a passably readable string of this ErrorSnapshot.
     * Used by String.valueOf(&#46;&#46;&#46;) when application-specific
     * transformation not provided.
     **/
    public String toString()
    {
        String comment = getComment();

        if (getThrown()!=null) {
            String what = getThrown().getMessage();
            return (comment!=null) ? (comment+" CAUSE: '"+what+"'") : what;

        } else if (comment!=null && comment.length()>0) {
            return comment;

        } else if (getLocation()!=null) {
            return getLocation().toString();
        }

        return uistrs().get("snapshot.fallback.msg",
                            getTargetName(),getTaskName());
    }


    /**
     * Returns the underlying exception's message if present. Otherwise
     * returns this snapshot's comment.
     **/
    public String stringFrom(Project p)
    {
        if (getThrown()!=null) {
            return getThrown().getMessage();
        }
        return getComment();
    }


    private String    m_Id;
    private String    m_targetName="", m_taskName="";
    private Location  m_location=Location.UNKNOWN_LOCATION;
    private Hashtable m_env;
    private String    m_comment="";
    private Exception m_thrown;
}

/* end-of-ErrorSnapshot.java */
