/**
 * $Id: Defaults.java 180 2007-03-15 12:56:38Z ssmc $
 * Copyright 2004-2005 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License (LGPL) as published
 * by the Free Software Foundation; either version 2.1 of the License, or (at your option)
 * any later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The GNU LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://www.jware.info                            EMAIL- inquiries@jware.info
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  org.apache.tools.ant.Project;

import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.helpers.Tk;
import  com.idaremedia.antx.parameters.MergeBehavior;

/**
 * Defaults for current application and/or iteration. Each default request is processed
 * as if it were the first; no response information is cached. Note that some defaults
 * can only exist at the global scope (no project-specific overrrides are permitted). To
 * alter the default values for global defaults, your application must set <em>either</em>
 * a fixture property (see {@linkplain AntXFixture}) or a global
 * <span class="src">System</span> property.
 * <p>
 * Most defaults are associated with a property. These properties are named based on
 * the current Iteration's {@linkplain Iteration#configId configuration prefix} so
 * Iteration subclasses do not need to subclass Defaults to alter the property's
 * prefix. If a default is documented as looking for "<span class="src">defaults.foo</span>",
 * this class will actually look for a property named
 * "<span class="src"><i>configId</i>.defaults.foo</span>" where <i>configId</i> is the
 * current Iteration's configuration prefix (this value is
 * <span class="src">jware.antx</span> for the standalone AntX distribution).
 *
 * @since     JWare/AntX 0.5
 * @author    ssmc, &copy;2004-2005 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version   0.5
 * @.safety   multiple
 * @.group    impl,helper
 * @see       com.idaremedia.antx.valueuri.info.DefaultsValueURIHandler
 **/

public class Defaults implements FixtureCore
{
    /**
     * Returns the name of the default properties prefix property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;property&#46;prefix</span>".
     **/
    public final String defaultPropertiesPrefixName()
    {
        return Iteration.configId()+".defaults.property.prefix";
    }


    /**
     * Returns the prefix prepended to default properties value URIs.
     * Looks at the "<span class="src">defaults.property.prefix</span>"
     * property. Returns <i>defaults&#46;</i> if not defined explicitly.
     * This is a global default.
     **/
    public String defaultPropertiesPrefix()
    {
        return AntXFixture.getProperty(defaultPropertiesPrefixName(),
                                       "defaults.");
    }



    /**
     * Returns the prefix prepended to default properties value URIs
     * looking into the given project for local overrides.
     * Looks at the "<span class="src">defaults.property.prefix</span>"
     * property.  Looks to {@linkplain #defaultPropertiesPrefix()}
     * for default setting.
     **/
    public String defaultPropertiesPrefix(Project project)
    {
        String s = project.getProperty(defaultPropertiesPrefixName());
        return Tk.isWhitespace(s) ? defaultPropertiesPrefix() : s;
    }



    /**
     * Returns the name of the main haltiferror property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;haltiferror&#46;flag</span>.
     **/
    public final String isHaltIfErrorName()
    {
        return Iteration.configId()+".defaults.haltiferror.flag";
    }



    /**
     * Returns the name of a feature-specific haltiferror property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;haltiferror&#46;<i>feature</i>&#46;flag</span>.
     **/
    public final String isHaltIfErrorName(String feature)
    {
        return Iteration.configId()+".defaults.haltiferror."+feature+".flag";
    }


    /**
     * Returns whether the application prefers a severe response to
     * problems when task-specific handling has not been defined.
     * Looks at the "<span class="src">defaults.haltiferror.flag</span>"
     * property. Returns <i>false</i> if default not set. This is a
     * global default.
     **/
    public boolean isHaltIfError()
    {
        String s = AntXFixture.getProperty(isHaltIfErrorName(),"no");
        return Tk.booleanFrom(s);
    }



    /**
     * Returns whether the application prefers a severe response to a
     * specific feature-bound problem when context-specific handling
     * has not been defined. Looks at the
     * "<span class="src">defaults.haltiferror.<i>feature</i>.flag</span>"
     * property where <i>feature</i> is the name of the application
     * feature (like "checksums"). Looks to {@linkplain #isHaltIfError()}
     * for default setting. This is a global default.
     * @param feature the feature's name (non-null)
     **/
    public boolean isHaltIfError(String feature)
    {
        String defaultYN= isHaltIfError() ? "yes" : "no";
        String s = AntXFixture.getProperty(isHaltIfErrorName(feature),
                                           defaultYN);
        return Tk.booleanFrom(s);
    }


    /**
     * Returns the name of the main assertions disabled property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;asserts&#46;flag</span>.
     **/
    public final String isAssertionsDisabledName()
    {
        return Iteration.configId()+".defaults.asserts.flag";
    }


    /**
     * Returns whether general assertions should be ignored in the
     * current execution iteration. Looks at the
     * "<span class="src">defaults.asserts.flag</span>" property.
     * Returns <i>true</i> if default not set. This is a global default.
     */
    public boolean isAssertionsDisabled()
    {
        String s = AntXFixture.getProperty(isAssertionsDisabledName(),
                                           "yes");
        return !Tk.booleanFrom(s);
    }



    /**
     * Returns the name of the delay file deletion property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;delaydelete&#46;flag</span>.
     **/
    public final String isTempFileDeletionDelayedName()
    {
        return Iteration.configId()+".defaults.delaydelete.flag";
    }



    /**
     * Returns whether the application wants transient scratch files
     * immediately deleted or not. (Delayed deletion allows diagnostics
     * on transient files.) Looks at the
     * "<span class="src">defaults.delaydelete.flag</span>" property.
     * Returns <i>false</i> if not set. This is a global default.
     **/
    public boolean isTempFileDeletionDelayed()
    {
        String s = AntXFixture.getProperty(isTempFileDeletionDelayedName(),
                                           "no");
        return Tk.booleanFrom(s);

    }



    /**
     * Like {@linkplain #isTempFileDeletionDelayed() isTempFileDeletionDelayed()}
     * but looks into the given project for local overrides.
     * @param project project that is first checked for property.
     **/
    public boolean isTempFileDeletionDelayed(Project project)
    {
        String s = project.getProperty(isTempFileDeletionDelayedName());
        return !Tk.isWhitespace(s) ? Tk.booleanFrom(s)
                    : isTempFileDeletionDelayed();
    }



    /**
     * Returns the name of the short locations string property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;shortlocations&#46;flag</span>.
     **/
    public final String isShortLocationsEnabledName()
    {
        return Iteration.configId()+".defaults.shortlocations.flag";
    }


    /**
     * Returns whether AntX can (pre)truncate really long location
     * strings in certain feedback messages. Looks at the
     * "<span class="src">defaults.shortlocations.flag</span>" property.
     * Returns <i>true</i> if default not set. This is a global default.
     */
    public boolean isShortLocationsEnabled()
    {
        String s = AntXFixture.getProperty(isShortLocationsEnabledName(),
                                           "yes");
        return Tk.booleanFrom(s);
    }



    /**
     * Returns the name of the fixture exclusion enabled property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;fixtureexcludes&#46;flag</span>.
     **/
    public final String isFixtureExcludesEnabledName()
    {
        return Iteration.configId()+".defaults.fixtureexcludes.flag";
    }



    /**
     * Returns whether fixture exclusions should be handled by default.
     * Looks at the
     * "<span class="src">defaults.fixtureexcludes.flag</span>" property.
     * Returns <i>false</i> if default not set. This is a global default.
     */
    public boolean isFixtureExcludesEnabled()
    {
        String s = AntXFixture.getProperty(isFixtureExcludesEnabledName(),
                                            "no");
        return Tk.booleanFrom(s);
    }


    /**
     * Returns the name of the passwords file property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;passwordfile</span>.
     **/
    public final String passwordFileLocationName()
    {
        return Iteration.configId()+".defaults.passwordfile";
    }



    /**
     * Returns the default (file) location of the current iteration's
     * services password file. The format of this file's contents is
     * iteration and/or application dependent. However, it is usually
     * a simple cleartext properties file (stored in a secured location
     * on file system). Looks at the
     * "<span class="src">defaults.passwordfile</span>" property in
     * the given project (first) and the global system properties.
     * Returns <i>null</i> if property not defined explicitly.
     * @param project [optional] project from which property should be
     *        first checked (will still look at system if not set)
     */
    public String passwordFileLocation(Project project)
    {
        return Tk.getTheProperty(project, passwordFileLocationName());
    }



    /**
     * Returns the name of the mergecontrol type property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;mergecontrol</span>.
     **/
    public final String mergeBehaviorName()
    {
        return Iteration.configId()+".defaults.mergecontrol";
    }


    /**
     * Returns the name of the feature-specific mergecontrol type property:
     * "<i>configid</i>&#46;<span class="src">defaults&#46;mergecontrol&#46;</span><i>feature</i>.
     **/
    public final String mergeBehaviorName(String feature)
    {
        return Iteration.configId()+".defaults.mergecontrol."+feature;
    }


    /**
     * Returns the default iteration configuration settings merge
     * behavior for the runtime. Defaults to {@linkplain MergeBehavior#FIRST}
     * if not set explicitly. Note that merge first means information
     * loaded later takes precedence. Looks at the
     * "<span class="src">defaults.mergecontrol</span>" property. This
     * is a global default.
     **/
    public MergeBehavior mergeBehavior()
    {
        return mergeBehavior(MergeBehavior.FIRST);
    }



    /**
     * Returns the default iteration configuration settings merge
     * behavior for the runtime using a caller-supplied
     * preference if not set explicitly. Looks at the
     * "<span class="src">defaults.mergecontrol</span>" property.
     * @param pref caller's merge preference (non-null)
     **/
    public MergeBehavior mergeBehavior(MergeBehavior pref)
    {
        String s = AntXFixture.getProperty(mergeBehaviorName(),
                                           pref.getValue());
        return MergeBehavior.from(s, pref);
    }



    /**
     * Returns the default iteration configuration settings merge
     * behavior for a specific feature. Defaults to a caller-supplied
     * preference if not set explicitly. Looks at the
     * "<span class="src">defaults.mergecontrol</span>" property for the
     * feature. This is a global default.
     * Example: <pre>
     *   Defaults defaults = Iteration.defaultdefaults();
     *   handling = defaults.mergeBehavior("aliases", defaults.mergeBehavior());
     *   switch(handling.getIndex()) {
     *      ...
     *   }
     * </pre>
     * @param feature the feature's name (non-null)
     * @param pref caller's merge preference (non-null)
     **/
    public MergeBehavior mergeBehavior(String feature, MergeBehavior pref)
    {
        String s = AntXFixture.getProperty(mergeBehaviorName(feature),
                                           pref.getValue());
        return MergeBehavior.from(s, pref);
    }



    /**
     * Initializes a new defaults instance. Iteration subclasses can
     * subclass our implementation to alter the default defaults or use
     * different properties.
     * @see Iteration#getDefaults
     **/
    protected Defaults()
    {
    }



    /**
     * Generic query interface for defaults by name. Used by utilities
     * like value URIs to extract defaults from Iteration subclasses.
     * @param field the field being queried (non-null, US-locale lowercased)
     * @param clnt call-controls (non-null)
     * @return field's current value as a string or <i>null</i> if no such field.
     * @since JWare/AntX 0.5
     **/
    public String valueFrom(String field, Requester clnt)
    {
        if ("assertions".equals(field)) {
            return String.valueOf(!isAssertionsDisabled());
        }
        if ("haltiferror".equals(field)) {
            return String.valueOf(isHaltIfError());
        }
        if (field!=null && field.startsWith("haltiferror.")) {
            String feature = field.substring("haltiferror.".length());
            return String.valueOf(isHaltIfError(feature));
        }
        if ("mergecontrol".equals(field)) {
            return String.valueOf(mergeBehavior());
        }
        if ("configid".equals(field)) {
            return Iteration.configId();
        }
        if ("noiselevel".equals(field)) {
            return NoiseLevel.getDefault(clnt.getProject()).toString();
        }
        if ("passwordfile".equals(field)) {
            return passwordFileLocation(clnt.getProject());
        }
        if ("delaydelete".equals(field)) {
            return String.valueOf(isTempFileDeletionDelayed(clnt.getProject()));
        }
        if ("defaults.prefix".equals(field)) {
            return defaultPropertiesPrefix(clnt.getProject());
        }
        if ("shortlocations".equals(field)) {
            return String.valueOf(isShortLocationsEnabled());
        }
        if ("fixtureexcludes".equals(field)) {
            return String.valueOf(isFixtureExcludesEnabled());
        }
        return null;
    }



    /**
     * Generic query interface for the name of a default attribute.
     * Used by utilities like value URIs to extract the name of the property
     * to set from Iteration subclasses.
     * @param field the field being queried (non-null, US-locale lowercased)
     * @param clnt call-controls (non-null)
     * @return field's property name or <i>null</i> if no such field.
     * @since JWare/AntX 0.5
     **/
    public String propertyFrom(String field, Requester clnt)
    {
        if (field==null) {
            return null;
        }
        if ("assertions".equals(field)) {
            return isAssertionsDisabledName();
        }
        if ("haltiferror".equals(field)) {
            return isHaltIfErrorName();
        }
        if (field.startsWith("haltiferror.")) {
            String feature = field.substring("haltiferror.".length());
            return isHaltIfErrorName(feature);
        }
        if ("mergecontrol".equals(field)) {
            return mergeBehaviorName();
        }
        if (field.startsWith("mergecontrol.")) {
            String feature = field.substring("mergecontrol.".length());
            return mergeBehaviorName(feature);
        }
        if ("noiselevel".equals(field)) {
            return AntX.DEFAULT_NOISELEVEL_PROP;
        }
        if ("passwordfile".equals(field)) {
            return passwordFileLocationName();
        }
        if ("delaydelete".equals(field)) {
            return isTempFileDeletionDelayedName();
        }
        if ("prefix".equals(field)) {
            return defaultPropertiesPrefixName();
        }
        if ("shortlocations".equals(field)) {
            return isShortLocationsEnabledName();
        }
        if ("fixtureexcludes".equals(field)) {
            return isFixtureExcludesEnabledName();
        }
        return null;
    }
}

/* end-of-Defaults.java */