/**
 * $Id: AssertableTask.java 187 2007-03-25 17:59:16Z ssmc $
 * Copyright 2002-2004,2007 iDare Media, Inc. All rights reserved.
 *
 * Originally written by iDare Media, Inc. for release into the public domain. This
 * library, source form and binary form, is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License as published by the
 * Free Software Foundation; either version 2.1 of the License, or (at your option) any
 * later version.<p>
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU LGPL (GNU Lesser General Public License) for more details.<p>
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite
 * 330, Boston, MA  02111-1307  USA. The LGPL can be found online at
 * http://www.fsf.org/copyleft/lesser.html<p>
 *
 * This product has been influenced by several projects within the open-source community.
 * The JWare developers wish to acknowledge the open-source community's support. For more
 * information regarding the open-source products used within JWare, please visit the
 * JWare website.
 *----------------------------------------------------------------------------------------*
 * WEBSITE- http://antxtras.sf.net/          EMAIL- jware[at]users[dot]sourceforge[dot]net
 *----------------------------------------------------------------------------------------*
 **/

package com.idaremedia.antx;

import  java.text.MessageFormat;

import  org.apache.tools.ant.BuildException;
import  org.apache.tools.ant.Project;
import  org.apache.tools.ant.Task;

import  com.idaremedia.apis.UIStringManager;

import  com.idaremedia.antx.apis.ProjectDependent;
import  com.idaremedia.antx.apis.Reconfigurable;
import  com.idaremedia.antx.apis.Requester;
import  com.idaremedia.antx.apis.ScriptLocatable;
import  com.idaremedia.antx.helpers.Tk;

/**
 * Extension of basic Ant <i>Task</i> that adds builtin assertions and support for
 * resource-bundle based messages.
 *
 * @since    JWare/AntX 0.1
 * @author   ssmc, &copy;2002-2004,2007 <a href="http://www.jware.info">iDare&nbsp;Media,&nbsp;Inc.</a>
 * @version  0.5.1
 * @.safety  guarded
 * @.group   impl,infra
 * @see      AssertableProjectComponent
 **/

public abstract class AssertableTask extends Task
    implements ProjectDependent, ScriptLocatable
{
    /**
     * Initializes new unlabeled task.
     **/
    protected AssertableTask()
    {
        super();
        Iam_="";
    }



    /**
     * Initializes new CV-labeled task.
     * @param iam CV-label (non-null)com.idare
     **/
    protected AssertableTask(String iam)
    {
        super();
        Iam_= Tk.cvlabelFrom(iam);
    }



    /**
     * Ensures this task's {@linkplain #initonce} method is called just
     * once. Sometimes Ant introspection mechanisms trigger mulitple calls
     * to init. This ensure that once-only initialization code is really
     * only called once.
     * @since JWare/AntX 0.5
     **/
    public void init()
    {
        super.init();
       
        if (!m_initedOnce) {
            initonce();
            m_initedOnce=true;
        }
    }



    /**
     * Initialization that must be done at most one time. Called by
     * {@linkplain #init} once even if init is itself called multiple
     * times.
     * @throws BuildException if unable to initialize required bits
     * @since JWare/AntX 0.5
     **/
    protected void initonce()
    {
    }



    /**
     * Resets this task's msgId attribute then reconfigures.
     **/
    public void reconfigure()
    {
        if (this instanceof Reconfigurable) {
            unconfigure();
        } else {
            String alert = uistrs().get("cv.redoable",Iam_,getTaskType());
            log(alert,Project.MSG_INFO);
            this.unconfigure();
        }
        super.reconfigure();
        m_configuredOnce=false;
    }
 
 
 
    /**
     * Resets this task as if never configured. Allows reconfigurable tasks
     * to be re-run many times. We just unset our <span class="src">msgId</span>
     * setting.
     * @since JWare/AntX 0.5
     */
    public void unconfigure()
    {
        m_msgId = null;
    }



    /**
     * Workaround NPE triggered problem in super.maybeConfigure since Ant 1&#46;7.
     * @since JWare/AntX 0.5.1
     */
    public void maybeConfigure()
    {
        if (!m_configuredOnce) {
            super.maybeConfigure();
            this.m_configuredOnce = true;
        }
    }



    /**
     * Ensures a child or contained item has the same owning target 
     * as this item if the child's is unassigned.
     * @param newitem child item (can be <i>null</i>)
     * @since JWare/AntX 0.5
     **/
    protected final void syncChildTargetIfNull(Object newitem)
    {
        //NB: necessary because of the way Ant's introspective creators
        //    work-- they do not sync parent/child target informaton!
        if (newitem instanceof Task) {
            Task t = (Task)newitem;
            if (t.getOwningTarget()==null) {
                t.setOwningTarget(getOwningTarget());
            }
        }
    }



    /**
     * Shortcut that returns this task's internal component UI
     * strings manager. Never returns <i>null</i>. By default returns
     * the current iteration's UI strings manager.
     * @see Iteration#uistrs
     **/
    public final UIStringManager uistrs()
    {
        return Iteration.uistrs();
    }


// ---------------------------------------------------------------------------------------
// (AntX) Universal Task Log Conversion (make events useful):
// ---------------------------------------------------------------------------------------

    /**
     * Indicate this task as source of logged message.
     **/
    public void log(String msg, int msgLevel)
    {
        if (getProject()!=null) {
            getProject().log(this,msg,msgLevel);
        } else {
            if (msgLevel >= Project.MSG_INFO) { //NB: works around bug in Task.log!
                System.err.println(msg);
            }
        }
    }

// ---------------------------------------------------------------------------------------
// (AntX) Universal Task Assertion Facilities:
// ---------------------------------------------------------------------------------------


    /**
     * Returns this task's CV-label. Never <i>null</i>.
     **/
    public final String cvlabel_()
    {
        return Iam_;
    }


    /**
     * Throws assertion error if pre-condtion is not met.
     * @param c pre-condition
     * @param msg [optional] failure message (if not met)
     * @throws IllegalArgumentException if condition not met
     **/
    protected final void require_(boolean c, String msg)
    {
        if (!c) {
            String ermsg = uistrs().get("cv.require",Iam_,msg);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalArgumentException(ermsg);
        }
    }


    /**
     * Throws assertion error if post-condition is not met. Used
     * for post-condition verification.
     * @param c post-condition
     * @param msg [optional] failure message (if not met)
     * @throws IllegalStateException if condition not met
     **/
    protected final void ensure_(boolean c, String msg)
    {
        if (!c) {
            String ermsg = uistrs().get("cv.ensure",Iam_,msg);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
    }


    /**
     * Throws assertion error if condition is not met. Used for
     * block and invariant verification.
     * @param c condition
     * @param msg [optional] failure message (if not met)
     * @throws IllegalStateException if condition not met
     **/
    protected final void verify_(boolean c, String msg)
    {
        if (!c) {
            String ermsg = uistrs().get("cv.verify",Iam_,msg);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
    }


    /**
     * Notes an unexpected but manageable problem has occured.
     * Just logs a warning by default.
     * @param t [optional] causing throwable
     * @param msg caller's additional (context) message
     **/
    protected final void unexpected_(Throwable t, String msg)
    {
        String ermsg = uistrs().get("cv.unexpected",Iam_,msg,t);
        log(ermsg, Project.MSG_WARN);
    }


    /**
     * Verifies we're in a live project (created from build process).
     * @throws IllegalStateException if not in project
     **/
    protected final void verifyInProject_(String calr)
    {
        if (getProject()==null) {
            String ermsg = uistrs().get("cv.verifyInP",Iam_,calr);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
    }


    /**
     * Verifies we're in a live target and project (created from
     * build process).
     * @throws IllegalStateException if not in target
     **/
    protected final void verifyInTarget_(String calr)
    {
        if (getOwningTarget()==null) {
            String ermsg = uistrs().get("cv.verifyInT",Iam_,calr);
            log(ermsg, Project.MSG_ERR);
            throw new IllegalStateException(ermsg);
        }
        verifyInProject_(calr);
    }


    /**
     * Called by '<i>execute</i>' on entry to verify that all required
     * options have been specified for correct execution of this task.
     * By default just verifies this task is associated with an enclosing
     * project.
     * @param calr calling method (usually 'execute' or 'run')
     * @throws BuildException if unable to execute
     * @throws IllegalStateException if improperly configured
     **/
    protected void verifyCanExecute_(String calr)
        throws BuildException
    {
        verifyInProject_(calr);
    }



    /**
     * Check whether a project property already exists and optionally
     * barfs if it does. Instance-level wrapper for generic
     * {@linkplain FixtureExaminer#checkIfProperty FixtureExaminer}
     * method.
     * @param property name of property to check
     * @param warn <i>true</i> if only issue a warning <em>otherwise can fail</em>
     * @return <i>true</i> if property already exists
     * @throws BuildException if property exists and isn't a warning
     * @since JWare/AntX 0.3
     **/
    protected final boolean checkIfProperty_(String property, boolean warn)
    {
        require_(property!=null,"chkProp- nonzro nam");

        return FixtureExaminer.checkIfProperty(getProject(),
            new Requester.ForComponent(this), property, warn);
    }



    /**
     * Check whether a project reference already exists and optionally
     * barfs if it does. Instance-level wrapper for generic
     * {@linkplain FixtureExaminer#checkIfReference FixtureExaminer}
     * method.
     * @param refid name of reference to check
     * @param warn <i>true</i> if only issue a warning <em>otherwise can fail</em>
     * @return <i>true</i> if reference already exists
     * @throws BuildException if reference exists and isn't a warning
     * @since JWare/AntX 0.4
     **/
    protected final boolean checkIfReference_(String refid, boolean warn)
    {
        require_(refid!=null,"chkRef- nonzro refid");

        return FixtureExaminer.checkIfReference(getProject(),
            new Requester.ForComponent(this), refid, warn);
    }


    /**
     * Returns the type-checked referenced object. Common utility that
     * generates resource bundle based messages if reference broken.
     * @param theProject the source project (null=> this task's project)
     * @param refid the referred-to thing's identifier (non-null)
     * @param requiredClass the required class of referred-to thing (non-null)
     * @throws BuildException if no such reference or object is not compatible
     * @since JWare/AntX 0.3
     **/
    public final Object getReferencedObject(Project theProject, String refid,
                                            Class requiredClass)
    {
        require_(refid!=null,"getRefObj- nonzro id");

        if (theProject==null) {
            theProject= getProject();
            verify_(theProject!=null,"getRefObj- hav project");
        }

        return FixtureExaminer.getReferencedObject(theProject, 
            new Requester.ForComponent(this), refid, requiredClass);
    }

// ---------------------------------------------------------------------------------------
// (AntX) Universal UI strings management:
// ---------------------------------------------------------------------------------------


    /**
     * Determine whether to check msgId against project properties
     * before checking UIStringManager. By default this task's msgid
     * is checked  against its project's properties.
     **/
    protected void setIgnoreProjectProperties(boolean ignored)
    {
        m_ignorePP= ignored ? Boolean.TRUE : Boolean.FALSE;
    }


    /**
     * Returns <i>true</i> if project properties are ignored in the
     * lookup of this task's messages.
     * @see AntX#PROPERTY_MSGIDS_PROP
     * @.impl Note the property is defined as positive if the build
     *         <em>allows</em> property-based msgids
     **/
    protected final boolean ignoreProjectProperties()
    {
        if (m_ignorePP==null) {
            if (getProject()!=null) {
                String s = getProject().getProperty(AntX.PROPERTY_MSGIDS_PROP);
                Boolean B = Tk.string2PosBool(s);
                if (B!=null) {
                    return !B.booleanValue();//NB: true=>allow, so ignore=>!allow
                }
            }
            return true;//the-default-setting
        }
        return m_ignorePP.booleanValue();
    }


    /**
     * Sets this task's default response message identifier.
     * This identifer is used with a {@linkplain UIStringManager} in
     * the ultimate concrete task subclass.
     **/
    public void setMsgId(String msgid)
    {
        m_msgId= msgid;
    }


    /**
     * Returns this task's default response message identifier.
     * Can return <i>null</i> if never set.
     **/
    public String getMsgId()
    {
        return m_msgId;
    }


    /**
     * Returns the localized message associated with given id. Looks for
     * messages using following steps:<ol>
     *   <li>Checks if msgid is a property with associated value; if
     *       value exists, it's returned. This step is skipped if this task's
     *       'ignoreProjectProperties' flag has been set to <i>true</i>.
     *   <li>Checks if there's a message in this task's default UI strings;
     *       if message exists, it's returned.
     *   <li>Returns msgid itself. This is useful instead of using a
     *       blank string since the msgid usually reflects some semantic
     *       meaning and it notifies the builder that the message is
     *       missing.
     * </ol>
     * @param msgId the message id (non-null)
     * @param getr retrieve UISM-based message (specific to API+args)
     * @see #getUISM
     * @see #getUISMArgs
     * @see #setIgnoreProjectProperties
     **/
    public String getMsg(String msgId, MsgGetter getr)
    {
        require_(msgId!=null && getr!=null,"getmsg- nonzro args");
        verifyInProject_("getMsg");

        //1. local-defn
        String msg = ignoreProjectProperties() ? null : getProject().getProperty(msgId);
        if (msg==null) {

            //2. uism-defn
            msg = getr.get(msgId);
            if (Tk.isWhitespace(msg)) {

                //3. fallback
                msg = msgId;
            }
        } else if (msg.indexOf("{")>=0) {//=> a MessageFormat template
            msg = MessageFormat.format(msg,getUISMArgs());
        }

        return msg;
    }


    /**
     * Same as {@linkplain #getMsg(String, MsgGetter) getMsg(String,MsgGetter)}
     * but with only the default message template variables inserted. Never
     * returns <i>null</i> but can return the empty string. The default formatting
     * automatically passes this task's name and build-file file/line information
     * to the MessageFormat.
     * @param getr the UIStringManager 'get' method pointer
     **/
    public String getMsg(MsgGetter getr)
    {
        String msgId = getMsgId();
        if (msgId==null) {
            return "";
        }
        return getMsg(msgId, getr);
    }


    /**
     * Shortcut {@linkplain #getMsg(MsgGetter) getMsg} that returns
     * the default msgid set on this task. Never returns <i>null</i> but
     * can return the empty string.
     * @see #getMsgId
     **/
    public String getMsg()
    {
        return getMsg(new MsgGetter() {
                public String get(String msgId) {
                    return getUISM().mget(msgId,m_UISMargs,"");
                }
            });
    }


    /**
     * Helper that returns this task's (class) default msg noise level
     * as if its never been specified. Subclass can change what's returned
     * to elevate or lower the default. Never returns <i>null</i>.
     **/
    public NoiseLevel getDefaultMsgLevel()
    {
        return NoiseLevel.getDefault(getProject());
    }


    /**
     * Initializes this task's default UISM arguments (taskname, location).
     **/
    private void getUISMDefaultArgs()
    {
        boolean shorten= Iteration.defaultdefaults().isShortLocationsEnabled();
        String locstr= Tk.shortStringFrom(shorten, getLocation());
        m_UISMargs= new Object[]{getTaskName(), locstr};
    }


    /**
     * Determines this task's target UIStringManager. Looks in the current
     * thread's iteration information for an activated UIStringManager. If
     * one not found, defaults to the currently installed root UIStringManager;
     * the default string manager is used only if no other string manager
     * is installed.
     * @see Iteration#stringManager
     **/
    public final UIStringManager getUISM()
    {
        synchronized (this) {
            if (m_UISMargs==null) {
                getUISMDefaultArgs();
            }
        }
        return Iteration.stringManager();
    }


    /**
     * Returns a copy of this task's always-on UIStringManager arguments.
     **/
    public final Object[] getUISMArgs()
    {
        synchronized (this) {
            if (m_UISMargs==null) {
                getUISMDefaultArgs();
            }
        }
        return new Object[] {m_UISMargs[0],m_UISMargs[1]};
    }


    /**
     * Helper that returns a new msg getter that uses UIStringManager
     * API for one additional non-standard argument. Task's name is always
     * for location {0} and task's declaration location is location {1}.
     * @param arg3 1st additional argument, template location {2}
     **/
    public final MsgGetter newMsgGetter(final String arg3)
    {
        return new MsgGetter() {
                public String get(String msgId) {
                    return getUISM().mget
                        (msgId, new Object[] {m_UISMargs[0],m_UISMargs[1],
                                              arg3});
                }
            };
    }


    /**
     * Helper that returns a new msg getter that uses UIStringManager
     * API for two additional arguments.
     * @param arg3 1st additional argument, template location {2}
     * @param arg4 2nd additional argument, template location {3}
     * @see #newMsgGetter(java.lang.String)
     **/
    public final MsgGetter newMsgGetter(final String arg3, final String arg4)
    {
        return new MsgGetter() {
                public String get(String msgId) {
                    return getUISM().mget
                        (msgId, new Object[] {m_UISMargs[0],m_UISMargs[1],
                                              arg3, arg4});
                }
            };
    }


    /**
     * Helper that returns a new msg getter that uses UIStringManager
     * API for three additional arguments.
     * @param arg3 1st additional argument, template location {2}
     * @param arg4 2nd additional argument, template location {3}
     * @param arg5 3rd additional argument, template location {4}
     * @see #newMsgGetter(java.lang.String)
     **/
    public final MsgGetter newMsgGetter(final String arg3, final String arg4,
                                        final String arg5)
    {
        return new MsgGetter() {
                public String get(String msgId) {
                    return getUISM().mget
                        (msgId, new Object[] {m_UISMargs[0],m_UISMargs[1],
                                              arg3, arg4, arg5});
                }
            };
    }


    /**
     * Returns instance of an <em>internal JWare/AntX</em> string or if
     * not found, the message identifier itself (as a placeholder).
     * @param msgId AntX message id (non-null)
     * @see #uistrs
     **/
    public final String getAntXMsg(String msgId)
    {
        require_(msgId!=null,"getAntXMsg- nonzro msgId");
        String msg = uistrs().get(msgId);
        if (Tk.isWhitespace(msg)) {
            msg = msgId;
        }
        return msg;
    }


    /**
     * Returns instance of a custom <em>internal JWare/AntX</em> string
     * or if not found the message identifier itself (as a placeholder).
     * @param msgId message id
     * @param arg1 additional argument, template location {0}
     **/
    public final String getAntXMsg(String msgId, String arg1)
    {
        require_(msgId!=null,"getAntXMsg- nonzro msgId");
        String msg = uistrs().get(msgId, arg1);
        if (Tk.isWhitespace(msg)) {
            msg = msgId;
        }
        return msg;
    }


    /**
     * Returns instance of a custom <em>internal JWare/AntX</em> string
     * or if not found the message identifier itself (as a placeholder).
     * @param msgId message id
     * @param arg1 1st additional argument, template location {0}
     * @param arg2 2nd additional argument, template location {1}

     **/
    public final String getAntXMsg(String msgId, String arg1, String arg2)
    {
        require_(msgId!=null,"getAntXMsg- nonzro msgId");
        String msg = uistrs().get(msgId, arg1,arg2);
        if (Tk.isWhitespace(msg)) {
            msg = msgId;
        }
        return msg;
    }


    /**
     * Returns instance of a custom <em>internal JWare/AntX</em> string
     * or if not found the message identifier itself (as a placeholder).
     * @param msgId message id
     * @param arg1 1st additional argument, template location {0}
     * @param arg2 2nd additional argument, template location {1}
     * @param arg3 3rd additional argument, template location {2}

     **/
    public final String getAntXMsg(String msgId, String arg1, String arg2, String arg3)
    {
        require_(msgId!=null,"getAntXMsg- nonzro msgId");
        String msg = uistrs().get(msgId, arg1,arg2,arg3);
        if (Tk.isWhitespace(msg)) {
            msg = msgId;
        }
        return msg;
    }


    private final String Iam_;
    private String m_msgId;
    private Object[] m_UISMargs;
    private Boolean m_ignorePP=null;//NB:true => don't ask Project.properties
    private boolean m_initedOnce;
    private boolean m_configuredOnce;//Ant 1.7 pains
}

/* end-of-AssertableTask.java */
