/*
 * Decompiled with CFR 0.152.
 */
package org.pageseeder.psml.md;

import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import org.eclipse.jdt.annotation.Nullable;
import org.pageseeder.psml.md.MarkdownOutputOptions;
import org.pageseeder.psml.md.MarkdownSerializer;
import org.pageseeder.psml.model.PSMLElement;
import org.pageseeder.psml.util.DiagnosticCollector;
import org.pageseeder.xmlwriter.XML;

class MarkdownTable {
    private static final int MAX_COLUMN_WIDTH = 32;
    private static final int MIN_COLUMN_WIDTH = 3;
    private static final int MAX_ROWS_PRETTY_SCAN = 20;
    private final PSMLElement table;
    private final String name;
    private final DiagnosticCollector collector;
    private final int columnCount;
    private final CellAlignment[] columnAlignments;
    private final CellStyle[] columnStyles;

    MarkdownTable(PSMLElement table, String name, DiagnosticCollector collector) {
        this.table = Objects.requireNonNull(table);
        this.name = name;
        this.collector = collector;
        this.columnCount = MarkdownTable.countColumnsForTable(table);
        this.columnAlignments = MarkdownTable.toColumnAlignments(this.table.getChildElements(PSMLElement.Name.COL), this.columnCount);
        this.columnStyles = MarkdownTable.toColumnStyles(this.table.getChildElements(PSMLElement.Name.COL), this.columnCount);
        if (PSMLElement.Name.TABLE != table.getElement()) {
            this.collector.error(name + " is not a table!");
        }
        if (table.getFirstChildElement(PSMLElement.Name.ROW) == null) {
            this.collector.error("No rows found in " + name);
        }
    }

    private static int countColumnsForTable(PSMLElement table) {
        int maxRowsChecked = 10;
        int count = 0;
        int rowsChecked = 0;
        for (PSMLElement col : table.getChildElements(PSMLElement.Name.COL)) {
            count += col.getAttributeOrElse("colspan", 1);
        }
        for (PSMLElement row : table.getChildElements(PSMLElement.Name.ROW)) {
            count = Math.max(count, MarkdownTable.countColumnsInRow(row));
            if (++rowsChecked < 10) continue;
            break;
        }
        return count;
    }

    private static int countColumnsInRow(PSMLElement row) {
        int count = 0;
        for (PSMLElement cell : row.getChildElements(PSMLElement.Name.CELL, PSMLElement.Name.HCELL)) {
            count += cell.getAttributeOrElse("colspan", 1);
        }
        return count;
    }

    private static CellAlignment[] toColumnAlignments(List<PSMLElement> columns, int columnCount) {
        CellAlignment[] alignments = new CellAlignment[columnCount];
        Arrays.fill((Object[])alignments, (Object)CellAlignment.NONE);
        int i = 0;
        for (PSMLElement col : columns) {
            CellAlignment alignment;
            String attribute = col.getAttribute("align");
            alignments[i] = alignment = MarkdownTable.toCellAlignment(attribute);
            if (++i < alignments.length) continue;
            break;
        }
        return alignments;
    }

    private static CellStyle[] toColumnStyles(List<PSMLElement> columns, int columnCount) {
        CellStyle[] styles = new CellStyle[columnCount];
        Arrays.fill((Object[])styles, (Object)CellStyle.NONE);
        int i = 0;
        for (PSMLElement col : columns) {
            CellStyle style;
            String attribute = col.getAttribute("part");
            styles[i] = style = MarkdownTable.toCellStyle(attribute);
            if (++i < styles.length) continue;
            break;
        }
        return styles;
    }

    public void format(Appendable out, MarkdownOutputOptions options) throws IOException {
        if (options.captions() && options.table() != MarkdownOutputOptions.TableFormat.HTML) {
            this.formatCaption(out);
        }
        if (options.table() == MarkdownOutputOptions.TableFormat.COMPACT) {
            this.formatCompact(out);
        } else if (options.table() == MarkdownOutputOptions.TableFormat.PRETTY) {
            this.formatPretty(out);
        } else if (options.table() == MarkdownOutputOptions.TableFormat.HTML) {
            this.formatHtml(out);
        } else if (options.table() == MarkdownOutputOptions.TableFormat.NORMALIZED) {
            this.formatNormalized(out);
        } else {
            this.collector.warn("Unrecognised table format: " + String.valueOf((Object)options.table()) + " - using compact format instead.");
            this.formatCompact(out);
        }
    }

    public void formatCaption(Appendable out) throws IOException {
        out.append("\n*").append(this.name);
        PSMLElement captionElement = this.table.getFirstChildElement(PSMLElement.Name.CAPTION);
        if (captionElement != null) {
            out.append(": ").append(captionElement.getText());
        }
        out.append("*\n\n");
    }

    public void formatCompact(Appendable out) throws IOException {
        List<PSMLElement> rows = this.table.getChildElements(PSMLElement.Name.ROW);
        int rowIndex = 0;
        for (PSMLElement row : rows) {
            String[] cells = this.toCells(row, rowIndex, false);
            this.formatCompactRow(out, cells);
            if (rowIndex == 0) {
                this.formatCompactHeaderLine(out);
            }
            ++rowIndex;
        }
        out.append('\n');
    }

    private void formatPretty(Appendable out) throws IOException {
        String[] cells;
        List<PSMLElement> rows = this.table.getChildElements(PSMLElement.Name.ROW);
        int[] columnWidths = new int[this.columnCount];
        Arrays.fill(columnWidths, 3);
        int rowIndex = 0;
        for (PSMLElement row : rows) {
            cells = this.toCells(row, rowIndex, false);
            for (int i = 0; i < cells.length; ++i) {
                columnWidths[i] = Math.max(columnWidths[i], Math.min(32, cells[i].length() + 2));
            }
            if (++rowIndex <= 20) continue;
            break;
        }
        rowIndex = 0;
        for (PSMLElement row : rows) {
            cells = this.toCells(row, rowIndex, false);
            this.formatPrettyRow(out, cells, columnWidths);
            if (rowIndex == 0) {
                this.formatPrettyHeaderLine(out, columnWidths);
            }
            ++rowIndex;
        }
        out.append('\n');
    }

    public void formatNormalized(Appendable out) throws IOException {
        List<PSMLElement> rows = this.table.getChildElements(PSMLElement.Name.ROW);
        int rowIndex = 0;
        for (PSMLElement row : rows) {
            String[] cells = this.toCells(row, rowIndex, true);
            this.formatCompactRow(out, cells);
            if (rowIndex == 0) {
                this.formatCompactHeaderLine(out);
            }
            ++rowIndex;
        }
        out.append('\n');
    }

    public void formatHtml(Appendable out) throws IOException {
        out.append("<table>\n");
        out.append("  <caption>").append(this.name);
        PSMLElement caption = this.table.getFirstChildElement(PSMLElement.Name.CAPTION);
        if (caption != null) {
            out.append(": ").append(XML.escape((String)caption.getText()));
        }
        out.append("</caption>\n");
        for (PSMLElement row : this.table.getChildElements(PSMLElement.Name.ROW)) {
            boolean isHeaderRow = "header".equals(row.getAttribute("part"));
            out.append("  <tr>\n");
            List<PSMLElement> cells = row.getChildElements(PSMLElement.Name.CELL, PSMLElement.Name.HCELL);
            int span = 0;
            for (int i = 0; i < cells.size(); ++i) {
                PSMLElement td = cells.get(i);
                String elementName = isHeaderRow || td.getElement() == PSMLElement.Name.HCELL ? "th" : "td";
                out.append("    <").append(elementName);
                if (this.columnAlignments[i + span] != CellAlignment.NONE) {
                    out.append(" style=\"text-align:").append(this.columnAlignments[i].toString().toLowerCase()).append(";\"");
                }
                if (td.getAttribute("colspan") != null) {
                    out.append(" colspan=\"").append(td.getAttribute("colspan")).append('\"');
                    span += td.getAttributeOrElse("colspan", 1) - 1;
                }
                if (td.getAttribute("rowspan") != null) {
                    out.append(" rowspan=\"").append(td.getAttribute("rowspan")).append('\"');
                }
                out.append('>').append(XML.escape((String)td.getText().trim())).append("</").append(elementName).append(">\n");
            }
            out.append("  </tr>\n");
        }
        out.append("</table>\n");
    }

    private void formatCompactRow(Appendable out, String[] cells) throws IOException {
        boolean firstCell = true;
        for (String cell : cells) {
            if (firstCell) {
                firstCell = false;
                out.append("| ");
            } else {
                out.append(' ');
            }
            out.append(cell).append(" |");
        }
        out.append('\n');
    }

    private void formatCompactHeaderLine(Appendable out) throws IOException {
        boolean firstCell = true;
        for (int i = 0; i < this.columnCount; ++i) {
            if (firstCell) {
                out.append('|');
                firstCell = false;
            }
            CellAlignment align = this.columnAlignments[i];
            out.append(MarkdownTable.headerLine(align, 3)).append('|');
        }
        out.append('\n');
    }

    private void formatPrettyRow(Appendable out, String[] cells, int[] columnWidths) throws IOException {
        boolean firstCell = true;
        for (int i = 0; i < cells.length; ++i) {
            if (firstCell) {
                firstCell = false;
                out.append("| ");
            } else {
                out.append(' ');
            }
            String cell = cells[i];
            CellAlignment align = this.columnAlignments[i];
            int pad = Math.max(columnWidths[i] - cell.length() - 2, 0);
            if (align == CellAlignment.RIGHT) {
                out.append(" ".repeat(pad));
                out.append(cell);
            } else {
                out.append(cell);
                out.append(" ".repeat(pad));
            }
            out.append(" |");
        }
        out.append('\n');
    }

    private void formatPrettyHeaderLine(Appendable out, int[] columnWidths) throws IOException {
        boolean firstCell = true;
        for (int i = 0; i < this.columnCount; ++i) {
            if (firstCell) {
                out.append('|');
                firstCell = false;
            }
            CellAlignment align = this.columnAlignments[i];
            out.append(MarkdownTable.headerLine(align, columnWidths[i])).append('|');
        }
        out.append('\n');
    }

    private String[] toCells(PSMLElement row, int rowIndex, boolean normalize) {
        String[] cells = new String[this.columnCount];
        CellStyle rowStyle = MarkdownTable.toCellStyle(row.getAttribute("part"));
        int i = 0;
        for (PSMLElement cell : row.getChildElements(PSMLElement.Name.CELL, PSMLElement.Name.HCELL)) {
            int colspan;
            CellStyle style = rowStyle != CellStyle.NONE ? rowStyle : this.columnStyles[i];
            cells[i] = MarkdownTable.toText(cell, rowIndex, style);
            int rowspan = cell.getAttributeOrElse("rowspan", 1);
            if (rowspan > 1) {
                this.collector.warn("Table cell " + (i + 1) + " at row " + (rowIndex + 1) + " has rowspan attribute, which is not supported in Markdown.");
            }
            if ((colspan = cell.getAttributeOrElse("colspan", 1)) > 1) {
                if (!normalize) {
                    this.collector.warn("Table cell " + (i + 1) + " at row " + (rowIndex + 1) + " has colspan attribute, which is not supported in Markdown.");
                }
                for (int j = 1; j < colspan; ++j) {
                    cells[i + j] = normalize ? cells[i] : "";
                }
            }
            if ((i += colspan) < cells.length) continue;
            break;
        }
        while (i < cells.length) {
            cells[i] = "";
            ++i;
        }
        return cells;
    }

    private static String toText(PSMLElement cell, int rowIndex, CellStyle style) {
        Object text = MarkdownSerializer.normalizeText(cell.getText().trim().replace("\n", "<br>"));
        if (style == CellStyle.BOLD && rowIndex > 0) {
            text = "**" + ((String)text).replace("*", "\\*") + "**";
        } else if (style == CellStyle.ITALIC) {
            text = "*" + ((String)text).replace("*", "\\*") + "*";
        }
        return text;
    }

    private static String headerLine(CellAlignment alignment, int width) {
        switch (alignment) {
            case LEFT: {
                return ":" + "-".repeat(width - 1);
            }
            case CENTER: {
                return ":" + "-".repeat(width - 2) + ":";
            }
            case RIGHT: {
                return "-".repeat(width - 1) + ":";
            }
        }
        return "-".repeat(width);
    }

    private static CellAlignment toCellAlignment(@Nullable String align) {
        if (align == null) {
            return CellAlignment.NONE;
        }
        switch (align) {
            case "left": {
                return CellAlignment.LEFT;
            }
            case "center": {
                return CellAlignment.CENTER;
            }
            case "right": {
                return CellAlignment.RIGHT;
            }
        }
        return CellAlignment.NONE;
    }

    private static CellStyle toCellStyle(@Nullable String part) {
        if (part == null) {
            return CellStyle.NONE;
        }
        if ("header".equals(part)) {
            return CellStyle.BOLD;
        }
        if ("footer".equals(part)) {
            return CellStyle.ITALIC;
        }
        return CellStyle.NONE;
    }

    private static enum CellStyle {
        NONE,
        BOLD,
        ITALIC;

    }

    private static enum CellAlignment {
        NONE,
        LEFT,
        CENTER,
        RIGHT;

    }
}

